/*** (c) STMicroelectronics ***************************************************
#
# PROJECT : ST7 USB LOW SPEED LIBRARY
#
# COMPILER : COSMIC / HIWARE
#
# VERSION : 4.21
#
# DESCRIPTION : USB lib core file
#
# HISTORY
# 08/25/00 -> Simplified INT_Usb() ISR to speed up exec time.
# V4.10	SetAddress modified
#           SetEP0RxStatus_IT and SetEP0TxStatus_IT created
#           STACK keyword (Cosmic) removed
# V4.11     Call to CallBack function MCU_Init added in INT_EndSusp.     
******************************************************************************/   

#include "USB_Rc.h"
#include "USB_Opts.h"
#include "Macro.h"
#include "USB_Def.h"
#include "Lib_Bits.h"
#include "USB_Var.h"
#include "USB.h"
#include "USB_Lib.h"

#ifdef USE_DEVICE_FIRMWARE_UPGRADE
#include "DFUCore.h"
#endif

#ifdef HIWARE
#include MAP_FILE
#pragma CODE_SEG USBLIB_ROM
#endif

#ifdef ENABLE_USER_STATUS_IN
extern void User_Status_In(void);  // Call-back function
#endif

#ifdef ENABLE_USER_STATUS_OUT
extern void User_Status_Out(void); // Call-back function
#endif

#ifdef INIT_APP_ON_END_SUSP
extern void User_MCU_Init(void);   // Call-back function
#endif

/********************************* Endpoint0 IN ***************************/

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetEP0TxTbc
INPUT/OUTPUT : Number of byte(s) to be sent
DESCRIPTION  : Write the byte number to send through USB
-----------------------------------------------------------------------------*/
void SetEP0TxTbc(Byte Tbc)
{
	USBEP0RA = (USBEP0RA & ~MASK_CNT) | Tbc;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetEP0TxStatus
INPUT/OUTPUT : status values
DESCRIPTION  : Write the status in EP0 Tx Register
-----------------------------------------------------------------------------*/
void SetEP0TxStatus(Byte Status)
{
	if (Status == VALID)
		USBEP0RA |= VALID;			// To avoid going through a DISABLE state
	else
		USBEP0RA = (USBEP0RA & ~STAT) | Status;         
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetEP0TxStatus_IT
INPUT/OUTPUT : status values
DESCRIPTION  : Write the status in EP0 Tx Register
-----------------------------------------------------------------------------*/
void SetEP0TxStatus_IT(Byte Status)
{
	if (Status == VALID)
		USBEP0RA |= VALID;			// To avoid going through a DISABLE state
	else
		USBEP0RA = (USBEP0RA & ~STAT) | Status;         
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : GetEP0TxStatus
INPUT/OUTPUT : Output the current Status value
DESCRIPTION  : Read the current EnpPoint0 TX Register Status
-----------------------------------------------------------------------------*/
Byte GetEP0TxStatus(void)
{
	return (USBEP0RA & STAT);
}


/********************************* Endpoint0 OUT ***************************/

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetEP0RxStatus
INPUT/OUTPUT : Status
DESCRIPTION  : Write the EP0 Rx status
-----------------------------------------------------------------------------*/                                  
void SetEP0RxStatus(Byte Status)
{
	if (Status == VALID)
		USBEP0RB |= VALID;			// To avoid going through a DISABLE state
	else
		USBEP0RB = (USBEP0RB & ~STAT) | Status;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetEP0RxStatus_IT
INPUT/OUTPUT : Status
DESCRIPTION  : Write the EP0 Rx status
-----------------------------------------------------------------------------*/                                  
void SetEP0RxStatus_IT(Byte Status)
{
	if (Status == VALID)
		USBEP0RB |= VALID;			// To avoid going through a DISABLE state
	else
		USBEP0RB = (USBEP0RB & ~STAT) | Status;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : GetEP0RxStatus
INPUT/OUTPUT : EP0 Rx Status
DESCRIPTION  : Output the current EnpPoint0 RX Status
-----------------------------------------------------------------------------*/
Byte GetEP0RxStatus(void)
{
	return (USBEP0RB & STAT);
}

#ifdef USE_ENDPOINT1_IN
/********************************* Endpoint1 IN ***************************/

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetEP1TxTbc
INPUT/OUTPUT : Number of byte(s) to be sent
DESCRIPTION  : Write the byte number to send through USB
-----------------------------------------------------------------------------*/
void SetEP1TxTbc(Byte Tbc)
{
	USBEP1RA = (USBEP1RA & ~MASK_CNT) | Tbc;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetEP1TxStatus
INPUT/OUTPUT : status values
DESCRIPTION  : Write the status in EP1 Tx Register
-----------------------------------------------------------------------------*/
void SetEP1TxStatus(Byte Status)
{
	if (Status == VALID)
		USBEP1RA |= VALID;			// To avoid going through a DISABLE state
	else
		USBEP1RA = (USBEP1RA & ~STAT) | Status;         
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : GetEP1TxStatus
INPUT/OUTPUT : Output the current Status value
DESCRIPTION  : Read the current EnpPoint1 TX Register Status
-----------------------------------------------------------------------------*/
Byte GetEP1TxStatus(void)
{
	return (USBEP1RA & STAT);
}

#endif 


#ifdef USE_ENDPOINT1_OUT
/********************************* Endpoint1 OUT ***************************/

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetEP1RxStatus
INPUT/OUTPUT : status
DESCRIPTION  : Write the EP1 Rx status
-----------------------------------------------------------------------------*/                                  
void SetEP1RxStatus(Byte Status)
{
	if (Status == VALID)
		USBEP1RB |= VALID;			// To avoid going through a DISABLE state
	else
		USBEP1RB = (USBEP1RB & ~STAT) | Status;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : GetEP1RxStatus
INPUT/OUTPUT : EP1 Rx Status
DESCRIPTION  : Output the current EnpPoint1 RX Status
-----------------------------------------------------------------------------*/
Byte GetEP1RxStatus(void)
{
	return (USBEP1RB & STAT);
}

#endif


#ifdef USE_ENDPOINT2_IN
/********************************* Endpoint2 IN ***************************/

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetEP2TxTbc
INPUT/OUTPUT : Number of byte(s) to be sent
DESCRIPTION  : Write the byte number to send through USB
-----------------------------------------------------------------------------*/
void SetEP2TxTbc(Byte Tbc)
{
	USBEP2RA = (USBEP2RA & ~MASK_CNT) | Tbc;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetEP2TxStatus
INPUT/OUTPUT : status values
DESCRIPTION  : Write the status in EP2 Tx Register
-----------------------------------------------------------------------------*/
void SetEP2TxStatus(Byte Status)
{
	if (Status == VALID)
		USBEP2RA |= VALID;			// To avoid going through a DISABLE state
	else
		USBEP2RA = (USBEP2RA & ~STAT) | Status;         
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : GetEP2TxStatus
INPUT/OUTPUT : Output the current Status value
DESCRIPTION  : Read the current EnpPoint2 TX Register Status
-----------------------------------------------------------------------------*/
Byte GetEP2TxStatus(void)
{
	return (USBEP2RA & STAT);
}

#endif


#ifdef USE_ENDPOINT2_OUT
/********************************* Endpoint2 OUT ***************************/

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetEP2RxStatus
INPUT/OUTPUT : status
DESCRIPTION  : Write the EP2 Rx status
-----------------------------------------------------------------------------*/                                  
void SetEP2RxStatus(Byte Status)
{
	if (Status == VALID)
		USBEP2RB |= VALID;			// To avoid going through a DISABLE state
	else
		USBEP2RB = (USBEP2RB & ~STAT) | Status;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : GetEP2RxStatus
INPUT/OUTPUT : EP2 Rx Status
DESCRIPTION  : Output the current EnpPoint2 RX Status
-----------------------------------------------------------------------------*/
Byte GetEP2RxStatus(void)
{
	return (USBEP2RB & STAT);
}

#endif

/********************************* MISCELLANEOUS ***************************/

#ifdef HIWARE
#pragma CODE_SEG USBLIB_ROM
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetAddress
INPUT/OUTPUT : None
DESCRIPTION  : Set the Device address after completion of the Status Stage
-----------------------------------------------------------------------------*/
void SetAddress(void)
{ 
	USBDADDR = USBwValue[0];			// Status In has been Acknowledged -> Set the device address.
	USBTransferStatus &= ~ADDRESS2SET;	// Clear ADDRESS2SET in USBTransferStatus.
	if (UsbInfo & CONFIGURED_STATE) return; // If configured just update the address
	if (USBwValue[0] != 0)
		UsbInfo = (UsbInfo & ~USB_STATE) | ADDRESS_STATE;	// Device has been moved to Address State.
	else
		UsbInfo = (UsbInfo & ~USB_STATE);					// Device has been moved back to Default State.
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetDmaAdd
INPUT/OUTPUT : DMA high and low bytes address values
DESCRIPTION  : Set the DMA buffers starting address
-----------------------------------------------------------------------------*/                                                  
void SetDmaAdd(Word Dmar)
{
	USBDMAR = Dmar/256;
	USBIDR = (Dmar%256) & 0xC0;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : CheckValidResume
INPUT/OUTPUT : None
DESCRIPTION  : Function called by INT_EndSusp ISR.
-----------------------------------------------------------------------------*/
void CheckValidResume(void)
{
#if 0
	Byte	j,k;

	j=0;
	for (k=100; k>0; k--)		// Loop for 400 us (or 800 us with OSCI = 12MHz)
	{// Test if K state or SE0 is valid
		if (((USBPIDR & JK_BUS_STATE) == K_STATE) || ((USBPIDR & SE0_BUS_STATE) == SE0))
			j++;
	}

	if (j > 90)			// is the signal stable ?
	{	/* Valid Resume -> Exit from Suspend State */
		USBLibStatus &= ~USB_SUSPEND;	// Inform app code that suspend is over.
	}
#else if
	USBLibStatus &= ~USB_SUSPEND;	// Inform app code that suspend is over.
#endif
}

/******************** USB INTERRUPTS SERVICE ROUTINES ************************/

/*-----------------------------------------------------------------------------

ROUTINE NAME : INT_EndSusp
DESCRIPTION  : USB End of Suspend mode interrupt routine.
				Called when host sends a resume signal
-----------------------------------------------------------------------------*/ 
#ifdef COSMIC
@interrupt void INT_EndSusp(void)
#endif
#ifdef HIWARE
#pragma TRAP_PROC SAVE_REGS
void INT_EndSusp(void)
#endif
{
	CheckValidResume();		// Check if valid resume and Reset suspend flags.
	USBISTR = ~Int_Esusp;	// Clear pending bit    
#ifdef INIT_APP_ON_END_SUSP
	User_MCU_Init();        // Call-back function used to initialize the MCU.
#endif	
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : INT_Usb
DESCRIPTION  : Comes from the USB cell.
-----------------------------------------------------------------------------*/ 
#ifdef COSMIC
@interrupt void INT_Usb(void)
#endif
#ifdef HIWARE
#pragma TRAP_PROC SAVE_REGS
void INT_Usb(void)
#endif
{// WARNING !! DO NOT CHANGE THE ORDER OF THE FOLLOWING STATEMENTS !!
	Byte	EndPointNumber, PacketID;

	if (USBISTR & Int_Reset)
	{
        UsbBusReset(); // USB Bus Reset Service routine.
        USBISTR = 0; // Clear USBISTR (USB Reset is the highest priority).
#ifdef USE_DEVICE_FIRMWARE_UPGRADE
        if (USBLibStatus2 & DFU_DETACH_RECEIVED) {
          DFUDeviceState = DFU_STATE_DFUIDLE;
          DFUDeviceStatus = DFU_STATUS_OK;
          #asm          
#ifdef HIWARE
            RSP;          
            JP DFU_Process;
#endif
#ifdef COSMIC
            XDEF _DFU_Process;
            RSP;          
            JP _DFU_Process;
#endif            
          #endasm          
        }
#endif
	}

	if (USBISTR & Int_Ctr)
	{	
		EndPointNumber = (USBIDR & MASK_EP)>>4;	// Get EndPoint number.
		PacketID = (USBPIDR & TP);				// Get Packet ID.

		if (EndPointNumber == 0)		// Test if Ctr IT occured on EP0
		{   // -> decode PacketID      
			if (PacketID == TP_SETUP)
			{
				UsbCtrFlag |= Int_Ctr_SETUP;	// Set Int_Ctr_SETUP flag
			}
			else
			if (PacketID == TP_IN) 
			{   
				if (USBTransferStatus & DATA_STAGE_IN)
				{	// Data Stage IN
					SetEP0RxStatus_IT(NAK);    		// set OUT direction to NACK
                    UsbCtrFlag |= Int_Ctr_IN;       // Set Int_Ctr_IN flag
				}
				else
				{	// Data Stage OUT or No Data Stage -> Status IN
					if (USBTransferStatus & ADDRESS2SET)
					{
						SetAddress();			// Address actually set here
					}
					SetEP0TxStatus_IT(STALL);	//  Next IN will be STALLed.
					SetEP0RxStatus_IT(STALL);	//  Next OUT will be STALLed.
#ifdef ENABLE_USER_STATUS_IN
					User_Status_In();		// Call-back function
#endif
				}					
			}
			else 
			{   // PacketID = TP_OUT
#ifdef USE_ENDPOINT0_OUT
				if (USBTransferStatus & DATA_STAGE_OUT)
				{	// Data stage OUT
					SetEP0TxStatus_IT(NAK);      			// set IN direction to NACK
                    			UsbCtrFlag |= Int_Ctr_OUT;  		    // Set Int_Ctr_OUT flag
					OUT_DataNumberEP0 = USBIDR & MASK_CNT;	// Save Number of received data.
				}
				else
#endif
				{	// No data stage (not supposed to happen!) or DATA_STAGE_IN (->status stage OUT)
					SetEP0TxStatus_IT(STALL);	//  Next IN will be STALLed.
					//SetEP0RxStatus_IT(STALL);	//  Next OUT will be ACKed.
#ifdef ENABLE_USER_STATUS_OUT
                			User_Status_Out();		// Call-back function                     
#endif
				}
			}
		}   // end of EP0 ISR
#ifdef USE_ENDPOINT1_OUT
		else if (EndPointNumber == 1)	// Test if Ctr IT occured on EP1
		{   
			if (PacketID == TP_OUT)
				OUT_DataNumberEP1 = USBIDR & MASK_CNT;		// Save Number of received data.
		}    // end of EP1 ISR
#endif
#ifdef USE_ENDPOINT2_OUT
		else if (EndPointNumber == 2)	// Test if Ctr IT occured on EP2
		{
			if (PacketID == TP_OUT)
				OUT_DataNumberEP2 = USBIDR & MASK_CNT;		// Save Number of received data.
		}   // end of EP2 ISR
#endif   
		USBISTR = ~Int_Ctr;					// Clear pending bit
	} // end of USB Correct Transfer (CTR) ISR

	if (USBISTR & Int_Susp)
	{                                
		USBCTLR = CtrlSusp;		// Set Suspend bit in USBCTLR to actually start logic suspend
		USBLibStatus |= USB_SUSPEND;	// Inform app code that suspend has started
		USBISTR = ~Int_Susp;			// Clear pending bit
	}
}

#ifdef HIWARE
#pragma CODE_SEG DEFAULT
#endif

/*** END OF FILE ***/