/*** (c) STMicroelectronics ***************************************************
#
# PROJECT : ST7 USB LOW SPEED EVALUATION KIT + DFU
#
# COMPILER : COSMIC / HIWARE
#
# VERSION : 1.1
#
# LIB VERSION : 4.21
#
# DESCRIPTION : Contains applet data processing
#
# HISTORY :
#								                                                              
******************************************************************************/

#include "Applet.h"
#include "Lib_Bits.h"
#include "Macro.h"
#include "User_Def.h"
#include "User_Var.h"
#include "USB_Opts.h" 
#include "USB_Var.h"
#include "USB_Def.h"
#include "USB_Lib.h"

#ifdef HIWARE
#include MAP_FILE
#pragma CODE_SEG USER_ROM
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : Command_LED
INPUT/OUTPUT : None
DESCRIPTION  : Toggle LED2 (PA7) according to Set Feature value
-----------------------------------------------------------------------------*/
void Command_LED (void)
{
	if (Applet_DataReceived[1] == 0)
		PADR |= 0x80;		//  If Reported Data =0 -> Switch OFF Led
	else
		PADR &= ~0x80;		//  If Reported Data =1 -> Switch ON Led
}
 
/*-----------------------------------------------------------------------------
ROUTINE NAME : Command_PWM
INPUT/OUTPUT : None
DESCRIPTION  : Modify Output Compare register value (PWM Duty Cycle) according to Set Feature
-----------------------------------------------------------------------------*/
void Command_PWM(void)
{
#ifdef ST7262
	PWMDCR0 = (5U * Applet_DataReceived[1]/2); // Change PWM Duty Cycle
#endif
#if defined(ST7263) || defined(ST7263B)
	TIMOC1LR = 100 - Applet_DataReceived[1];	// OLVL2 output level -> Change PWM Duty Cycle
#endif
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : Read_ADC
INPUT/OUTPUT : None
DESCRIPTION  : Send ADC Value to Host through EP1
-----------------------------------------------------------------------------*/
void Read_ADC (void)
{   
	if (ValBit(ADCCSR,7))				// has conversion completed ?
	{
		Applet_DataToSend[0] = 3;				// Store Report Id Number in DemokitReportIN[0].
#ifdef ST7262
#ifdef DFU_NEW_FIRMWARE
        Applet_DataToSend[1] = ADCDRMSB;
#else
        Applet_DataToSend[1] = (ADCDRMSB & 0xC0);
#endif
#endif                
#if defined(ST7263) || defined(ST7263B)
#ifdef DFU_NEW_FIRMWARE
        Applet_DataToSend[1] = ADCDR;
#else
        Applet_DataToSend[1] = (ADCDR & 0xC0);
#endif
#endif		
		IN_NbBytesToSend = 2;
		Applet_Data_Ready = TRUE;
	}
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : Read_BUTTON
INPUT/OUTPUT : None
DESCRIPTION  : Send SW1 Button value (PA5) to the Host when pressed or released, through EP1
-----------------------------------------------------------------------------*/
void Read_BUTTON(void)
{
 
	if ((!(PADR & 0x20)) && (!(Button_Pressed)))		
	{	// Button has been pressed
		Button_Pressed = TRUE;          // state = pushed
		Applet_DataToSend[0] = 2;	// Report Id Number
		Applet_DataToSend[1] = 1;	// Data to Report
		IN_NbBytesToSend = 2;
		Applet_Data_Ready = TRUE;       // Ask to send new button state
	}

	if ((PADR & 0x20) && (Button_Pressed))
	{   // Button has been released                            
		Button_Pressed = FALSE;         // state = released
		Applet_DataToSend[0] = 2;	// Report Id Number
		Applet_DataToSend[1] = 0;	// Data to Report
		IN_NbBytesToSend = 2;
		Applet_Data_Ready = TRUE;       // Ask to send new button state
	}
	
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : StopConsumption
INPUT/OUTPUT : None
DESCRIPTION  : Prepare application before enter in Suspend mode
-----------------------------------------------------------------------------*/
void StopConsumption(void)
{ 
  // Optional board modification : Remove W1, connect PB2 to W1B1
  PBDR &= ~0x04; // Cut ADC resistor 5v
  // Disable ADC  
  ADCCSR = 0;
#ifdef ST7262
  // Disable PWM outputs
  PWMCR = 0;
  // Disable PWM
  ARTCSR = 0;
#endif
#if defined(ST7263) || defined(ST7263B)
  // Disable Timer
  TIMCR1 = 0;
  TIMCR2 = 0;
  TIMSR |= 0x04;
#endif
#ifdef ST7262
  // Switch-off LEDs and Set I/O in Output 0
  PADR = 0xC0;
  PADDR = 0xC6;
  PBDR = 0xC2;
  PBDDR = 0xFF;
  PCDR = 0;
  PCDDR = 0xFF;
  PDDR = 0;
  PDDDR = 0xFF;
#endif
#if defined(ST7263) || defined(ST7263B)
  // Switch-off LEDs and Set I/O in Output 0
  PADR = 0xC0;
  PADDR = 0xC6;
  PBDR = 0xC2;
  PBDDR = 0xFF;
  PCDR = 0;
  PCDDR = 0xFF;
#endif
}

#ifdef USE_REPORT5
/*-----------------------------------------------------------------------------
ROUTINE NAME : IN_SendReport5ToPC
INPUT/OUTPUT : None
DESCRIPTION  : Send HID Report #5 data to PC through EP1 (Interrupt)               
-----------------------------------------------------------------------------*/
void IN_SendReport5ToPC(void)
{

unsigned char i;
 
  if (IN_CountPass == 0) { // First Pass
    IN_StillToSend = REPORT5_LENGTH + 1; // Add one byte for the ReportID
  }
  
  if (IN_StillToSend > 8) {
    IN_CountPass++;
    if (IN_CountPass == 1) { // First transfer
      Applet_DataToSend[0] = 5; // ReportID mandatory at first pass
      Applet_DataToSend[1] = HIBYTE(IN_tempo); // Dummy value
      Applet_DataToSend[2] = LOBYTE(IN_tempo); // Dummy value
      Applet_DataToSend[3] = IN_CountPass; // Dummy value
      Applet_DataToSend[4] = IN_StillToSend; // Dummy value
      Applet_DataToSend[5] = 0xAA; // Dummy value
      Applet_DataToSend[6] = 0xBB; // Dummy value
      Applet_DataToSend[7] = 0xCC; // Dummy value
    }
    else {
      Applet_DataToSend[0] = 5; // Dummy value
      Applet_DataToSend[1] = HIBYTE(IN_tempo); // Dummy value
      Applet_DataToSend[2] = LOBYTE(IN_tempo); // Dummy value
      Applet_DataToSend[3] = IN_CountPass; // Dummy value
      Applet_DataToSend[4] = IN_StillToSend; // Dummy value
      Applet_DataToSend[5] = 0xDD; // Dummy value
      Applet_DataToSend[6] = 0xEE; // Dummy value
      Applet_DataToSend[7] = 0xFF; // Dummy value 
    }  		      	   
    IN_NbBytesToSend = 8; // Maximum number of data to send at a time on EP1
    IN_StillToSend -= 8; // Calculate the remaining bytes still to send    		       
  }
  else {
    if (IN_CountPass == 0) { // Only one pass
      Applet_DataToSend[0] = 5; // ReportID mandatory
      for (i=1; i<IN_StillToSend; i++) Applet_DataToSend[i] = LOBYTE(IN_tempo); // Dummy values
    }
    else {
      Applet_DataToSend[0] = 5; // Dummy value
      for (i=1; i<IN_StillToSend; i++) Applet_DataToSend[i] = LOBYTE(IN_tempo); // Dummy value		        
      IN_CountPass = 0; // Reset for next transfer
    }
    IN_NbBytesToSend = IN_StillToSend;
    IN_StillToSend = 0; // No more data to send  
  }
    
  Applet_Data_Ready = TRUE;
		        	
}
#endif

#ifdef USE_REPORT5
/*-----------------------------------------------------------------------------
ROUTINE NAME : REQ_SendReport5ToPC
INPUT/OUTPUT : None
DESCRIPTION  : Send HID Report #5 data to PC through EP0 (i.e. GetFeature report)
-----------------------------------------------------------------------------*/
void REQ_SendReport5ToPC(void) {

unsigned char i;

          if (Test_EP_Ready(0, EP_IN)) { // If EndPoint is ready to send data
          
	    if (USBwLength > 8) { // Multiple data transfer	      
	      if (USBDataXferStatus == FIRST_DATA) {
	        Applet_DataToSend[0] = 5; // ReportID mandatory in first byte
	        for (i=1; i<8; i++) Applet_DataToSend[i] = i; // Dummy values
	        USBDataXferStatus = STILL_DATA; // There are still data to send
	      }
	      else for (i=0; i<8; i++) Applet_DataToSend[i] = i + 2; // Dummy values  	    
	      Write_EP_Buffer(0, Applet_DataToSend, 8);
	      Set_EP_Ready(0, EP_IN, 8);
	      USBwLength -= 8;
	      if (USBwLength == 0) USBDataXferStatus = LAST_DATA; // No more data to send  	      	     	   
	    }
	    else { // USBwLength <= 8
	      if (USBDataXferStatus == FIRST_DATA) { // Only one transfer
	        Applet_DataToSend[0] = 5; // ReportID mandatory in first byte
	        for (i=1; i<USBwLength; i++) Applet_DataToSend[i] = i; // Dummy values	        
	        Write_EP_Buffer(0, Applet_DataToSend, USBwLength);
	        Set_EP_Ready(0, EP_IN, USBwLength);
		USBLibStatus &= ~APP_REQUEST; // Clear APP_REQUEST flag since there is no more data to send	        
	      }
	      else if (USBDataXferStatus != LAST_DATA) { // If it is not the last transfer
	        for (i=0; i<USBwLength; i++) Applet_DataToSend[i] = i + 8; // Dummy values
	        Write_EP_Buffer(0, Applet_DataToSend, USBwLength);
	        Set_EP_Ready(0, EP_IN, USBwLength);
	        USBDataXferStatus = LAST_DATA;
	      }  	      	      	     	      	      
	    }
	    
	  }	    
}
#endif

#ifdef USE_REPORT5
/*-----------------------------------------------------------------------------
ROUTINE NAME : REQ_GetReport5FromPC
INPUT/OUTPUT : None
DESCRIPTION  : Get Report #5 data coming from PC (i.e. SetFeature)
-----------------------------------------------------------------------------*/
void REQ_GetReport5FromPC(void) {

unsigned char i;
static unsigned int ReportIndex;

  	if (Test_EP_Ready(0, EP_OUT)) { // If EndPoint has received data
  	
   	  if (USBwLength > 8) {
   	    if (USBDataXferStatus == FIRST_DATA) { // First data transfer
   	      for (i=1; i<8; i++) Report5DataFromPC[i-1] = Applet_DataReceived[i]; // Save 7 bytes (ReportID is skipped)
              USBDataXferStatus = STILL_DATA; // There are still data to receive
              ReportIndex = 7; // Initialize index to store received data
            }
   	    else {
   	      for (i=0; i<8; i++) Report5DataFromPC[ReportIndex+i] = Applet_DataReceived[i]; // Save 8 bytes
              ReportIndex += 8;
            }
   	    Set_EP_Ready(0, EP_OUT, 0);	   
  	    USBwLength -= 8; // Update remaining bytes to be received
  	    if (USBwLength == 0) USBDataXferStatus = LAST_DATA; // No more data to receive      	    
   	  }
  	  else { // USBwLength <= 8
  	    if (USBDataXferStatus == FIRST_DATA) {
    	      for (i=1; i<USBwLength; i++) Report5DataFromPC[i-1] = Applet_DataReceived[i]; // Save bytes (ReportID is skipped)
	      USBLibStatus &= ~APP_REQUEST;
     	    }
     	    else if (USBwLength > 0) {
     	      for (i=0; i<USBwLength; i++) Report5DataFromPC[ReportIndex+i] = Applet_DataReceived[i]; // Save remaining bytes
	      Set_EP_Ready(0, EP_OUT, 0);	      
     	      USBDataXferStatus = LAST_DATA; // No more data to receive
     	      USBwLength = 0;
     	      ReportIndex = 0;
     	    }
     	  }
     	    
  	}  	
}
#endif

#ifdef HIWARE
#pragma CODE_SEG DEFAULT
#endif
 
/*** END OF FILE ***/