/*** (c) STMicroelectronics ***************************************************
#
# PROJECT : ST7 USB LOW SPEED EVALUATION KIT + DFU
#
# COMPILER : COSMIC / HIWARE
#
# VERSION :
#
# DESCRIPTION : Device Firmware Upgrade protocol file
#
******************************************************************************/

#include "DFUProtocol.h"
#include "DFUCore.h"
#include "Lib_Bits.h"  
#include "USB_Var.h"
#include "USB_Def.h"
#include "USB_Lib.h"
#include "Macro.h"
#include "HDFlash_macros.h"
#include "DFUDescript.h"

#ifdef HIWARE
#include MAP_FILE
#endif

//-------------------
// Protocol Variables
//-------------------
Byte DFUAction;
Byte DFUVpp12V;  // Used to test if Vpp is equal to 12V or not
Byte *DFUpFlash; // Pointer to Flash memory
Byte HDFlashUnLocked; // Used to test if FCSR register is locked or not
                                
#ifdef HIWARE                               
#pragma CODE_SEG DFU_PRTCL
#endif

#ifdef COSMIC
#pragma section (DFU_PRTCL)
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : VbusCheck
INPUT/OUTPUT : None/Byte
DESCRIPTION  : Switch USB on or off depending the presence of Vbus.
RETURN VALUE : 0 = USB cable is removed, 1 = USB cable is plugged
-----------------------------------------------------------------------------*/
Byte VbusCheck(void)
{
  if (ValBit(PBDR, 5)) { // USB cable is plugged
    if (ValBit(USBCTLR, 2)) Init_USB_HW(); // if Regulator is OFF --> Switch USB ON
    return 1;
  }
  else { // USB cable is removed
    if (!ValBit(USBCTLR, 2)) Disable_USB_HW(); // if Regulator is ON --> Switch USB OFF      
    return 0;
  }
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DFU_Init_User
INPUT/OUTPUT : None
DESCRIPTION  : Called by DFU_Init.
-----------------------------------------------------------------------------*/
void DFU_Init_User(void)
{
  PBDR = 0x02;
  PBDDR = 0x02; // PB1 OUT for Vpp control
  DFUPollTimeOutH = 0; // 5ms
  DFUPollTimeOutL = 5;
  DFUAction = DFU_ACTION_NONE;
  HDFlashUnLocked = FALSE;
  DFUVpp12V = FALSE;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : HDFlashProcess
INPUT/OUTPUT : None
DESCRIPTION  : HDFlash commands processing.
-----------------------------------------------------------------------------*/
void HDFlashProcess(void)
{

  Byte i;

  if (VbusCheck() == 0) {
    WDGCR = 0x80; // MCU Reset 
  }
 
  if (DFUAction & DFU_ACTION_START) {          

    DisableInterrupts;
    
    if (DFUVpp12V == FALSE) SetVpp12V(); // Set Vpp to 12V if not already done

    //---------------
    // Erase Sector 1
    //---------------
    if (DFUAction & DFU_ACTION_ERASE_S1) {                        
      HDFlashEraseSector(1, 1);
      if (HDFlashStatus == HDFLASH_ERASE_OK) {
        DFUDeviceStatus = DFU_STATUS_OK;
        DFUDeviceState = DFU_STATE_DFUDNLOADSYNC;
      }
      else {
        DFUDeviceStatus = DFU_STATUS_ERRERASE;
        DFUDeviceState = DFU_STATE_DFUERROR;
        goto Erase_stop; // If Sector 1 erasing is fail we don't erase Sector 2
      }
    }         

    //---------------
    // Erase Sector 2
    //---------------
    if (DFUAction & DFU_ACTION_ERASE_S2) {                        
      HDFlashEraseSector(2, 1);
      if (HDFlashStatus == HDFLASH_ERASE_OK) {
        DFUDeviceStatus = DFU_STATUS_OK;
        DFUDeviceState = DFU_STATE_DFUDNLOADSYNC;
      }
      else {
        DFUDeviceStatus = DFU_STATUS_ERRERASE;
        DFUDeviceState = DFU_STATE_DFUERROR;
      }
    }

    // Jump here if Sector 1 erasing failed
    Erase_stop:
          
    //--------------------------
    // Programmation (Byte mode)
    //--------------------------
    if (DFUAction & DFU_ACTION_PROGRAM) {
      for (i=0; i<8; i++) { // Program 8 bytes, one at a time.
         HDFlashWriteByte(DFUDataReceived[i], (unsigned int)(DFUpFlash+i), 1);
         if (HDFlashStatus != HDFLASH_BYTEPROG_OK) {
           DFUDeviceStatus = DFU_STATUS_ERRPROG;
           DFUDeviceState = DFU_STATE_DFUERROR;
           goto Prog_stop;
         }
      }
      DFUDeviceStatus = DFU_STATUS_OK;
      DFUDeviceState = DFU_STATE_DFUDNLOADSYNC;                   
    }          
    // Jump here if Programmation failed
    Prog_stop:

    EnableInterrupts;

    DFUPollTimeOutH = 0; // 5ms
    DFUPollTimeOutL = 5;
    DFUAction = DFU_ACTION_NONE;

  } // End Action Start
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetVpp12V
INPUT/OUTPUT : None
DESCRIPTION  : Set Vpp to 12V to erase or program the HDFlash.
-----------------------------------------------------------------------------*/
void SetVpp12V(void)
{
  PBDR &= ~0x02; // Connect Vcc on ST662A
  // Temporization to wait stabilization of ST662A Vout
  // 4ms @ 4MHz, 2ms @ 8MHz
  #asm
    LD    X,#10;
    CLR   A;
  delay1:
    INC	  A;
    JRNE  delay1;
    DEC   X;
    JRNE  delay1; 
  #endasm  
  DFUVpp12V = TRUE;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetVpp5V
INPUT/OUTPUT : None
DESCRIPTION  : Set Vpp to 5V.
-----------------------------------------------------------------------------*/
void SetVpp5V(void)
{
  PBDR |= 0x02; // Disconnect Vcc on ST662A
  DFUVpp12V = FALSE;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : UnlockFCSR
INPUT/OUTPUT : None
DESCRIPTION  : Unlock FCSR register. RASS keys are received from Host.
-----------------------------------------------------------------------------*/
void UnlockFCSR(void)
{
  if (HDFlashUnLocked == FALSE) {
    // RASS keys are sent by the Host
    RASS_Disable(DFUDataReceived[0], DFUDataReceived[1]);
    HDFlashUnLocked = TRUE;
  }
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DFU_Abort_User
INPUT/OUTPUT : None
DESCRIPTION  : Called by DFU_Abort.
-----------------------------------------------------------------------------*/
void DFU_Abort_User(void)
{ 
  DFUAction = DFU_ACTION_NONE;
  SetVpp5V();
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DFU_ClearStatus_User
INPUT/OUTPUT : None
DESCRIPTION  : Called by DFU_ClearStatus.
-----------------------------------------------------------------------------*/
void DFU_ClearStatus_User(void)
{
  DFUPollTimeOutH = 0; // 5ms
  DFUPollTimeOutL = 5;
  DFUAction = DFU_ACTION_NONE;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DFU_ManifestWaitReset_User
INPUT/OUTPUT : None
DESCRIPTION  : Called by DFU_GetStatus.
               Last action before to enter in dfuMANIFEST-WAIT-RESET state.
-----------------------------------------------------------------------------*/
void DFU_ManifestWaitReset_User(void)
{ 
  WDGCR = 0x80; // MCU Reset
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DFU_UpLoad_User
INPUT/OUTPUT : None
DESCRIPTION  : Called by DFU_UpLoad.
-----------------------------------------------------------------------------*/
void DFU_UpLoad_User(void)
{

  unsigned char i, LengthTmp;

  LengthTmp = min(USBwLength, 8); // Maximum 8 data per UpLoad request      

  for (i=0; i<LengthTmp; i++) {
    DFUDataToSend[i] = *(Byte *)(MAKEWORD(USBwValue[1],USBwValue[0]) + i); // Protocol
  }

  Write_EP_Buffer(0, DFUDataToSend, LengthTmp);

  Set_EP_Ready(0, EP_IN, LengthTmp);

  if ((DFUDeviceState == DFU_STATE_DFUUPLOADIDLE) && (USBwLength > 8)) // Protocol
    DFUDeviceState = DFU_STATE_DFUIDLE; // End of UpLoad (Short Frame)
  else
    DFUDeviceState = DFU_STATE_DFUUPLOADIDLE;

  DFUDeviceStatus = DFU_STATUS_OK;
  USBLibStatus &= ~APP_REQUEST;

}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DFU_Download_User
INPUT/OUTPUT : None
DESCRIPTION  : Called by DFU_Download.
-----------------------------------------------------------------------------*/
void DFU_Download_User(void)
{

  Read_EP_Buffer(0, DFUDataReceived); // Save received data on EP0

  //---------------------
  // Check Protocol Codes
  //---------------------
  
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  // Note for PollTimeOut : erasing time are given according to
  // fCPU = 8MHz for ST7262-16K and fCPU = 4MHz for ST7263B-16K
  //++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

  if (MAKEWORD(USBwValue[1],USBwValue[0]) == DFU_BLKNUM_ERASE_S1) { // Protocol
    DFUAction = DFU_ACTION_ERASE_S1;
#ifdef ST7262
    DFUPollTimeOutH = 0x09; // 2500ms = 09C4h
    DFUPollTimeOutL = 0xC4;
#endif
#ifdef ST7263B
    DFUPollTimeOutH = 0x0F; // 4000ms = 0FA0h
    DFUPollTimeOutL = 0xA0;
#endif
    UnlockFCSR();
    return;
  }

  if (MAKEWORD(USBwValue[1],USBwValue[0]) == DFU_BLKNUM_ERASE_S2) { // Protocol
    DFUAction = DFU_ACTION_ERASE_S2;
#ifdef ST7262
    DFUPollTimeOutH = 0x09; // 2500ms = 09C4h
    DFUPollTimeOutL = 0xC4;
#endif
#ifdef ST7263B
    DFUPollTimeOutH = 0x13; // 5000ms = 1388h
    DFUPollTimeOutL = 0x88;
#endif
    UnlockFCSR();
    return;
  }

  DFUAction = DFU_ACTION_PROGRAM;
  DFUPollTimeOutH = 0; // 5ms
  DFUPollTimeOutL = 5;
  DFUpFlash = (Byte *)(MAKEWORD(USBwValue[1],USBwValue[0])); // Protocol

}

#ifdef HIWARE
#pragma CODE_SEG DEFAULT
#endif

#ifdef COSMIC
#pragma section ()
#endif

/*** END OF FILE ***/