/*** (c) STMicroelectronics ***************************************************
#
# PROJECT : ST7 USB LOW SPEED LIBRARY
#
# COMPILER : COSMIC / HIWARE
#
# VERSION :
#
# DESCRIPTION : USB lib core file  
#
******************************************************************************/   
   
#include "USB.h"   
#include "Macro.h"
#include "Lib_Bits.h"
#include "USB_Def.h"
#include "USB_Var.h"                                                                                                            
#include "USB_Rc.h"
#include "USB_Lib.h" 
#include "DFUCore.h"
#include "DFUDescript.h"

#ifdef HIWARE
#include MAP_FILE                               
#pragma CODE_SEG USBLIB_ROM
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : UsbBusReset
INPUT/OUTPUT : None
DESCRIPTION  : Reset registers and variables associated to USB and Disable com on EP0
				Called when a USB Bus Reset is received from Host.
-----------------------------------------------------------------------------*/
void UsbBusReset (void)
{

	// USB Variables initialization

	UsbReport[0] = 0;					// Initialize UsbReport variable
	UsbReport[1] = 0;					// Initialize UsbReport variable
	ConfigValue = 0;					// Default Configuration.
	UsbCtrFlag = 0;			// Clear CTR Int. flags
	USBTransferStatus = 0;			// Clear USB Control status flags.
	UsbInfo	= DEFAULT_STATE;			// Add =0|Config=0|Device awake   
	USBLibStatus &= 0x07;			// Reset lib status bits. Keep product selection.

        USBDataXferStatus = FIRST_DATA;           // Used to send/receive more than 8 data
    
	OUT_DataNumberEP0 = 0;				// Reset Number of received data on EP0 when OUT Ctr.
    
	DeviceStatusInfo |= SELF_POW; // Self Powered Device                                          

	// Endpoint0 initialization. First Transaction --> SETUP ONLY EXPECTED ON EP0
	// All other EPs are disable 
	SetEP0TxStatus_IT(STALL);			//  Next IN will be stalled.
	SetEP0RxStatus_IT(STALL);			//  Next OUT will be stalled.

	// Initialize USBIMR
	USBIMR = 0xA6;					// set USB interrupt mask. Enable SUSP|CTR|ESUSP|RESET

}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetConfiguration
INPUT/OUTPUT : None
DESCRIPTION  : Assign configuration value sent by Host & reset Stalled EPs.
-----------------------------------------------------------------------------*/
void SetConfiguration(void)
{	// Valid request in ADDRESS_STATE and CONFIGURED_STATE

     if ((USBwValue[0] == 0) || (USBwValue[0] == DFU_Config_Desc[5]))
	{
		ConfigValue = USBwValue[0];
		if (ConfigValue != 0)
		{
			UsbInfo = (UsbInfo & ~USB_STATE) | CONFIGURED_STATE;	// Device is now configured
			USBLibStatus |= USB_CONFIGURED;
		}
		else
		{
			UsbInfo = (UsbInfo & ~USB_STATE) | ADDRESS_STATE;	// Device has been set back to Address State
			USBLibStatus &= ~USB_CONFIGURED;
		}
	}
	else
		RequestError();				// Inappropriate Request
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : GetStatus
INPUT/OUTPUT : None
DESCRIPTION  : Build status report into UsbReport[] global variable.
-----------------------------------------------------------------------------*/
void GetStatus()
{	// REM: Request Valid only on EP0 while in ADDRESS STATE

	switch (USBbmRequestType & RECIPIENT)
	{

	case REINTERFACE :	
          if (UsbInfo & CONFIGURED_STATE) {
            if (USBwIndex == 0) { // Interface 0
              UsbReport[0] = 0; // Send a 0 Word according to spec.
              UsbReport[1] = 0; // Send a 0 Word according to spec.
            }
            else { // Non existing interface
              RequestError();
            }
          }
          else { // Request only valid in the Configurated state
            RequestError();
          }
	break;		

	case REDEVICE :
		{
			UsbReport[0] = DeviceStatusInfo & 0x03;
		}
		break; // REM: USBwIndex = 0.

	case REENDPOINT : 
		if (USBwIndex == 0x80)	// EP0 IN
		{
			UsbReport[0] = (DeviceStatusInfo>>2) & 0x01;	// Send Status for EP0 
		}
		else if (USBwIndex == 0x00)	// EP0 OUT
		{
			UsbReport[0] = (DeviceStatusInfo>>2) & 0x01;	// Send Status for EP0 
		}
		else if (!(UsbInfo & CONFIGURED_STATE))	// Valid request on EP0 ONLY, while in ADDRESS STATE.
		{
			RequestError();		// Inappropriate Request. 
		}

		// Device is in CONFIGURED_STATE
		else
		{
			RequestError();		// Inappropriate Request. Endpoint does not exist.
		}
		break;				// End of Recipient = Endpoint

	default : 
		RequestError();		// Inappropriate Request -> Recipient does not exist.
	}						// End of switch Recipient
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetupStage
INPUT/OUTPUT : None
DESCRIPTION  : Process the Setup Stage token in Control Transfer.
-----------------------------------------------------------------------------*/
void SetupStage(void)
{

	Word ActualDescLength, wLength_temp;

	USBbmRequestType = EP0OutBuffer[0];
	USBbRequest      = EP0OutBuffer[1];
	USBwValue[0]	 = EP0OutBuffer[2];
	USBwValue[1]	 = EP0OutBuffer[3];
	USBwIndex	 = MAKEWORD(EP0OutBuffer[5],EP0OutBuffer[4]);
        USBwLength	 = MAKEWORD(EP0OutBuffer[7],EP0OutBuffer[6]);

      USBDataXferStatus = FIRST_DATA; // Used to send/receive more than 8 data

	if (USBwLength == 0) // wLength equals to 0 means there is no Data stage.		
	{	// Requests without data stages are processed here.              
	
		USBTransferStatus = NO_DATA_STAGE;	// A Status In will follow the Setup Stage

		switch (USBbmRequestType & TYPE)	
		{
		case STANDARD :
			switch (USBbRequest)
			{
			case SET_ADDRESS:
			  if (USBwValue[0] > 0x7F)
			  { // USB address is coded on 7 bits
		 	    RequestError(); // Inappropriate Request
			  }
			  else
			  {
			    USBTransferStatus |= ADDRESS2SET;
			  }
			  return; // Address actually assigned at completion of Status Stage
			case SET_CONFIGURATION: SetConfiguration(); 
						return;
			default: RequestError(); // Inappropriate Request
				 return;
			}

		case CLASS :
			switch (USBbRequest)
			{
			case DFU_CLRSTATUS :
			case DFU_ABORT :
			case DFU_DNLOAD :
				USBLibStatus |= APP_REQUEST;
				return;
			default : RequestError(); // Inappropriate Request
				  return;
			}
		default: RequestError(); // Inappropriate Request
			 return;
		}		// End of Request Type selection
	}		// End of NO Data Stage Section 

	else
	{// Following Requests require a Data Stage
		if (USBbmRequestType & EP_IN)
		{// a 'data stage in' will follow the 'Setup stage'
			USBTransferStatus = DATA_STAGE_IN;	// A Data Stage IN will follow the Setup Stage
												// -> We need to select and prepare Data to Send
			switch (USBbmRequestType & TYPE)	
			{
			case STANDARD:
				switch (USBbRequest)
				{
				case GET_DESCRIPTOR :
					switch (USBwValue[1])								// Descriptor Type Selection
					{

					case DEVICE_DESC :	
						DescAddBasePointer = DFU_Device_Desc;	// Device Desc Base Pointer
                           			ActualDescLength = DFU_Device_Desc[0]; // Device Desc Length
						break;	

					case STRING_DESC : 
						switch (USBwValue[0])							// String Descriptor Offset 
						{
						case 0 :				
                                			DescAddBasePointer = DFU_Lang_String;	// String Desc Base Pointer
							ActualDescLength = DFU_Lang_String[0];
							break;

						case INDEX_MANUFACTURER :
                                			DescAddBasePointer = DFU_Manuf_String;	// String Desc Base Pointer
							ActualDescLength = DFU_Manuf_String[0];
							break;

						case INDEX_NAME :				
							DescAddBasePointer = DFU_Name_String;	// String Desc Base Pointer
							ActualDescLength = DFU_Name_String[0]; 
							break;
						default :
							RequestError();		// Inappropriate Request
							return;
						}				// End of switch(USBwValue[0])
						USBwIndex = 0;	// To start reading at the begining of the Descriptor.
						break;			// End of String Desc section.

					case CONFIGURATION_DESC :	
							DescAddBasePointer = DFU_Config_Desc;	// Configuration Desc Base Pointer
							ActualDescLength = MAKEWORD(DFU_Config_Desc[3],DFU_Config_Desc[2]); 
						break;

					default : 
						RequestError();		// Inappropriate Request
						return;
					}	// End of Descriptor Type Selection
					wLength_temp = USBwLength;
					USBwLength = min(wLength_temp,ActualDescLength);	// Total Number of bytes to be sent to the host
					return;			// End of Case GET_DESCRIPTOR preprocess
				case GET_CONFIGURATION :	USBwLength = 1; // Return 1 byte maximum (spec)
								UsbReport[0] = ConfigValue;
								break;

				case GET_STATUS :		USBwLength = 2; // Return 2 bytes maximum (spec)
								GetStatus(); 
								break;

				default:			RequestError();		// Inappropriate Request
								return;
				}
				break;						// End of STANDARD Requests

			case CLASS:
				switch (USBbRequest)
				{
				case GET_REPORT : USBLibStatus |= APP_REQUEST;
						  break;
				case DFU_UPLOAD :
				case DFU_GETSTATUS :
				case DFU_GETSTATE :
					USBLibStatus |= APP_REQUEST;			
					break;
				default: RequestError(); // Inappropriate Request
					 return;
				}
				break;					// End of CLASS Requests
			default: RequestError();		// Inappropriate Request
				 return;
			}		// End of Request Type Selection
		}		// End of Data Stage IN Preprocess.

		else
		{// a 'Data Stage OUT' will follow the 'Setup Stage'
			USBTransferStatus = DATA_STAGE_OUT;		// Data Stage OUT will follow the Setup Stage.
													// -> Read and process data that will be received
			switch (USBbmRequestType & TYPE)
			{
			case CLASS :
				switch (USBbRequest)
				{
				case DFU_DNLOAD : return;	// Data will be processed on next OUT Token.
				default : RequestError(); // Inappropriate Request    
					  return;
				}
			default : RequestError(); // Inappropriate Request
			} // end of request type selection
		}	// End of 'Data Stage Out' section
	}	// End of 'Data Stage' section	
}	// End of SetupStage()  function

/*-----------------------------------------------------------------------------
ROUTINE NAME : DataStageIn
INPUT/OUTPUT : None
DESCRIPTION  : Process 'Data Stage IN' in Control Transfer (Ep0).
-----------------------------------------------------------------------------*/ 
void DataStageIn(void)
{                   
	if (USBwLength == 8)
	{// Packet size is  8 bytes -> Need to send an empty data packet to a potential IN.
		USBTransferStatus |= ONE_MORE;
	} 
	LengthToSend = min(8,USBwLength);	// Bytes to be sent at next IN Token
	USBwLength -= LengthToSend;			// Update USBwLength (Remaining bytes to be sent)
	if (USBbRequest == GET_DESCRIPTOR)  
	{
		DataToSend = &DescAddBasePointer[USBwIndex];
		USBwIndex += LengthToSend;				            // Update Current Descriptor Pointer
	}
	else  
	{
		DataToSend = UsbReport;
	}
	Write_EP_Buffer(0,DataToSend,LengthToSend);	// Load DataToSend in EP0 DMA buff.
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DataStageOut
INPUT/OUTPUT : None
DESCRIPTION  : Process 'Data stage OUT' in Control Transfer (Ep0).
-----------------------------------------------------------------------------*/
void DataStageOut(void)
{     
	// There is no standard request requiring a 'data stage out' 
	// -> all data stage out requests must be processed at application level
	USBLibStatus |= APP_REQUEST;
}
                                                
/*-----------------------------------------------------------------------------
ROUTINE NAME : UsbTransfer
INPUT/OUTPUT : None
DESCRIPTION  : Process a Correct Transfer Interrupt on EP0 (Enumeration process).
-----------------------------------------------------------------------------*/
void UsbTransfer(Byte TokenType)
{         
	if (TokenType & Int_Ctr_SETUP)
	{ 
		SetupStage();			// Decode the Setup Token
		if (GetEP0RxStatus() == STALL)
		{
			return;			// A request error occured in SetupStage().
		}
		else if (USBLibStatus & APP_REQUEST)
		{// This Request (CLASS or VENDOR) is handle in Handle_App_Requests()
			return;	// Will advance to STATUS OUT when returning from Handle_App_Requests()
		}
		else if (USBTransferStatus & NO_DATA_STAGE)
		{// STATUS IN expected -> Stall OUT token
			LengthToSend = 0;					// Transmit counter will be reset to 0.
			USBTransferStatus |= SET_TX0_VALID;	// Will send empty data on next IN.
			SetEP0RxStatus(STALL);              // STALL non expected OUT token.
		}
		else if (USBTransferStatus & DATA_STAGE_IN)	
		{// Initialise Data IN Transfer 
			DataStageIn();
			USBTransferStatus |= SET_RX0_VALID;			// Status OUT may occur.
			USBTransferStatus |= SET_TX0_VALID;			// Data will be sent on next IN.
		}
		else  //case USBTransferStatus = DATA_STAGE_OUT
		{// Initialise Data OUT Transfer 
			LengthToSend = 0;			// Transmit counter will be reset to 0 = Status IN
			USBTransferStatus |= SET_TX0_VALID;		// Case STATUS IN occurs.
			USBTransferStatus |= SET_RX0_VALID;		// Data will be received on next OUT Token.
		}
	} 
	else     
	if (!(USBLibStatus & APP_REQUEST))
	{// Data stage process of a Standard request
		if  (TokenType & Int_Ctr_IN)
		{// Data Stage In progress
			if (USBwLength == 0)
			{
				if (USBTransferStatus & ONE_MORE)
				{// Data length is a multiple of 8 -> Send an empty data packet on potential next IN.
					LengthToSend = 0;						// Transmit counter will be reset to 0.
					USBTransferStatus |= SET_TX0_VALID;		// Empty data will be sent on next IN.		
					USBTransferStatus |= SET_RX0_VALID;		// Status OUT may occur.
					USBTransferStatus &= ~ONE_MORE;			// Reset ONE_MORE flag
				}
				else
				{
					SetEP0TxStatus(STALL);					// All data sent -> Next IN will be stalled.
					USBTransferStatus |= SET_RX0_VALID;		// Status OUT may occur.
				}
			}
			else
			{// USBwLength != 0 -> Prepare next data transmission.   
				DataStageIn();								// Update DataToSend & LengthToSend
				USBTransferStatus |= SET_RX0_VALID;			// Status OUT may occur.
				USBTransferStatus |= SET_TX0_VALID;			// Data will be sent on next IN.
			}
		}  // end data stage IN
	   	else
		{  // Data Stage Out Progress
			DataStageOut();				// Process Received Data
		}
	}// End of standard request data stage processing
}// End of USBTransfer()


/*-----------------------------------------------------------------------------
ROUTINE NAME : RequestError
INPUT/OUTPUT : None
DESCRIPTION  : Stall next Data or Status Stage. 
				Function called when an irrelevant request has been received.
-----------------------------------------------------------------------------*/
void RequestError(void)
{	// RequestError is called only during enumeration -> STALL EP0 both dir.
	SetEP0RxStatus(STALL);
	SetEP0TxStatus(STALL);
}

#ifdef HIWARE                               
#pragma CODE_SEG DEFAULT
#endif

/*** END OF FILE ***/
