/*** (c) STMicroelectronics ***************************************************
#
# PROJECT : ST7 USB LOW SPEED LIBRARY
#
# COMPILER : COSMIC / HIWARE
#
# VERSION :
#
# DESCRIPTION : USB lib interface file
#               
******************************************************************************/   

#include "USB_Lib.h"
#include "Macro.h"
#include "USB_Def.h"
#include "Lib_Bits.h"
#include "USB_Var.h"
#include "USB_Rc.h"
#include "USB.h"

#ifdef HIWARE
#include MAP_FILE
#pragma CODE_SEG USBLIB_ROM
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : Init_USB_HW
INPUT/OUTPUT : None
DESCRIPTION  : Function called by user to switch on the USB cell.  
				-> Connect the peripheral to the bus
-----------------------------------------------------------------------------*/
void Init_USB_HW (void)
{
	/* Reset Usb variables used in Handle_USB_Events() */
	UsbCtrFlag = 0;				// Clear CTR Int. flags
	ConfigValue = 0;			// Default Configuration. 
	/* Initialize DMA engine and USB Cell */
	SetDmaAdd((Word) &EP0OutBuffer[0]);	// set DMA buffer starting address 
	USBIMR = 0;				// Mask all USB interrupts
	USBCTLR = 0;				// Switch the USB Voltage Regulator ON	
	USBISTR = 0;				// Clear all USB interrupts flags
	USBIMR = 0x86;				// Enable SUSP, ESUSP and RESET USB interrupts only
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : Disable_USB_HW
INPUT/OUTPUT : None
DESCRIPTION  : Function called by user to switch off the USB cell. 
				-> Disconnect the peripheral from the bus
-----------------------------------------------------------------------------*/
void Disable_USB_HW (void)
{
	USBIMR = 0;           // Mask all USB interrupts
	USBCTLR = 0x06;       // Switch the USB Voltage Regulator OFF
	USBISTR = 0;          // Clear all USB interrupts flags
	UsbCtrFlag = 0;       // Clear CTR Int. flags
	USBLibStatus &= 0x07; // Reset lib status bits. Keep product selection.
}


/*-----------------------------------------------------------------------------
ROUTINE NAME : Handle_USB_Events
INPUT/OUTPUT : None
DESCRIPTION  : Handle all USB Interrupt and process STANDARD requests.
-----------------------------------------------------------------------------*/
void Handle_USB_Events(void)
{	
	if (UsbCtrFlag != 0)  
	{	
		// One USB transfer interrupt occured -> Process it here
		UsbTransfer(UsbCtrFlag);	// Service USB CTR IT
		UsbCtrFlag = 0;				// Clear flag

		// No more CTR IT are pending ->  re-enable transmission/reception on EP0.
		if (USBTransferStatus & SET_TX0_VALID)
		{	// Request to enable transmission on EP0 -> Answer to next IN token.
			Set_EP_Ready(0,EP_IN,LengthToSend);
			USBTransferStatus &= ~SET_TX0_VALID;
		}
		if (USBTransferStatus & SET_RX0_VALID)
		{	// Request to enable reception on EP0 -> ACKnowledge next OUT token.
			Set_EP_Ready(0,EP_OUT,0);
			USBTransferStatus &= ~SET_RX0_VALID;
		}
	}
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : Enable_STATUS_Stage
INPUT/OUTPUT : None
DESCRIPTION  : Function called by application once APP_REQUESTS processing
				is done in Handle_APP_Requests()
				-> Allow ST7 to ACK or STALL incoming Status Stage
-----------------------------------------------------------------------------*/
void Enable_STATUS_Stage(void)
{
	if (USBLibStatus & USB_STALL)
	{
		RequestError();
		USBLibStatus &= ~USB_STALL;     // Clear flag
		USBLibStatus &= ~APP_REQUEST;	// Clear APP_REQUEST flag
	}
	else 
	{
		if (USBTransferStatus & NO_DATA_STAGE)
		{
			SetEP0RxStatus(STALL);		// STALL ANY EXTRA OUT  
			Set_EP_Ready(0,EP_IN,0);	// ACK STATUS IN
		}
		else if (USBTransferStatus & DATA_STAGE_IN)
		{
			if (!(USBLibStatus & APP_REQUEST))
			{// "data stage in" is over -> Advance to "Stall extra IN"
				USBwLength = 0;	// will STALL ANY EXTRA IN (cf. UsbTransfer()) 
			}			
			Set_EP_Ready(0,EP_OUT,0);	// ACK STATUS OUT
		}
		else if (USBTransferStatus & DATA_STAGE_OUT)
		{
			if (!(USBLibStatus & APP_REQUEST))
			{// data stage out is over -> No more data expected
				SetEP0RxStatus(STALL);	// STALL ANY EXTRA OUT  
			}			
			Set_EP_Ready(0,EP_IN,0);	// ACK STATUS IN
		}
	}
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : Set_EP_Ready
INPUT/OUTPUT : Endpoint#, EP dir, Data Payload
DESCRIPTION  : Update Transmit byte counter and Set EPTx to VALID, or Set EPRx to VALID
				REM: Length is ignored when direction=EP_OUT
-----------------------------------------------------------------------------*/
void Set_EP_Ready(Byte Endpoint, Byte Direction, Byte Length)
{
		if (Direction==EP_IN)
		{
			SetEP0TxTbc(Length);                                                       
			SetEP0TxStatus(VALID);				// Enable EP0 for next IN token.
		}
		else
		{
			SetEP0RxStatus(VALID);				// Enable EP0 for next OUT token.
		}
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : Write_EP_Buffer
INPUT/OUTPUT : Endpoint#, Data Source Pointer, Number of bytes to transmit
DESCRIPTION  : Write data into the USB DMA RAM Transmit buffer of selected EP
-----------------------------------------------------------------------------*/
void Write_EP_Buffer (Byte Endpoint, Byte *Data, Byte DataLength)
{ 
	Byte i;
	for (i=0; i<DataLength; i++) EP0InBuffer[i] = *(Data + i);
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : Read_EP_Buffer
INPUT/OUTPUT : Endpoint#, Data Destination pointer
DESCRIPTION  : Read data from the USB DMA Receive buffer into *DataDestination
-----------------------------------------------------------------------------*/
void Read_EP_Buffer (Byte Endpoint, Byte *DataDestination)
{ 
	Byte i;
	for (i=0; i<OUT_DataNumberEP0; i++) *(DataDestination+i) = EP0OutBuffer[i];
}

#ifdef HIWARE
#pragma CODE_SEG DEFAULT
#endif

/*** END OF FILE ***/