/*** (c) STMicroelectronics ***************************************************
#
# PROJECT : ST7 USB LOW SPEED LIBRARY
#
# COMPILER : COSMIC / HIWARE
#
# VERSION :
#
# DESCRIPTION : USB lib core file
#
******************************************************************************/   

#include "USB_Rc.h"
#include "Macro.h"
#include "USB_Def.h"
#include "Lib_Bits.h"
#include "USB_Var.h"
#include "USB.h"
#include "USB_Lib.h"
#include "DFUCore.h"

#ifdef HIWARE
#include MAP_FILE
#pragma CODE_SEG USBLIB_ROM
#endif

extern void User_Status_Out(void); // Call-back function

/********************************* Endpoint0 IN ***************************/

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetEP0TxTbc
INPUT/OUTPUT : Number of byte(s) to be sent
DESCRIPTION  : Write the byte number to send through USB
-----------------------------------------------------------------------------*/
void SetEP0TxTbc(Byte Tbc)
{
	USBEP0RA = (USBEP0RA & ~MASK_CNT) | Tbc;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetEP0TxStatus
INPUT/OUTPUT : status values
DESCRIPTION  : Write the status in EP0 Tx Register
-----------------------------------------------------------------------------*/
void SetEP0TxStatus(Byte Status)
{
	if (Status == VALID)
		USBEP0RA |= VALID;			// To avoid going through a DISABLE state
	else
		USBEP0RA = (USBEP0RA & ~STAT) | Status;         
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetEP0TxStatus_IT
INPUT/OUTPUT : status values
DESCRIPTION  : Write the status in EP0 Tx Register
-----------------------------------------------------------------------------*/
void SetEP0TxStatus_IT(Byte Status)
{
	if (Status == VALID)
		USBEP0RA |= VALID;			// To avoid going through a DISABLE state
	else
		USBEP0RA = (USBEP0RA & ~STAT) | Status;         
}


/********************************* Endpoint0 OUT ***************************/

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetEP0RxStatus
INPUT/OUTPUT : Status
DESCRIPTION  : Write the EP0 Rx status
-----------------------------------------------------------------------------*/                                  
void SetEP0RxStatus(Byte Status)
{
	if (Status == VALID)
		USBEP0RB |= VALID;			// To avoid going through a DISABLE state
	else
		USBEP0RB = (USBEP0RB & ~STAT) | Status;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetEP0RxStatus_IT
INPUT/OUTPUT : Status
DESCRIPTION  : Write the EP0 Rx status
-----------------------------------------------------------------------------*/                                  
void SetEP0RxStatus_IT(Byte Status)
{
	if (Status == VALID)
		USBEP0RB |= VALID;			// To avoid going through a DISABLE state
	else
		USBEP0RB = (USBEP0RB & ~STAT) | Status;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : GetEP0RxStatus
INPUT/OUTPUT : EP0 Rx Status
DESCRIPTION  : Output the current EnpPoint0 RX Status
-----------------------------------------------------------------------------*/
Byte GetEP0RxStatus(void)
{
	return (USBEP0RB & STAT);
}

/********************************* MISCELLANEOUS ***************************/

#ifdef HIWARE
#pragma CODE_SEG USBLIB_ROM
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetAddress
INPUT/OUTPUT : None
DESCRIPTION  : Set the Device address after completion of the Status Stage
-----------------------------------------------------------------------------*/
void SetAddress(void)
{ 
	USBDADDR = USBwValue[0];			// Status In has been Acknowledged -> Set the device address.
	USBTransferStatus &= ~ADDRESS2SET;	// Clear ADDRESS2SET in USBTransferStatus.
	if (UsbInfo & CONFIGURED_STATE) return; // If configured just update the address
	if (USBwValue[0] != 0)
		UsbInfo = (UsbInfo & ~USB_STATE) | ADDRESS_STATE;	// Device has been moved to Address State.
	else
		UsbInfo = (UsbInfo & ~USB_STATE);					// Device has been moved back to Default State.
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SetDmaAdd
INPUT/OUTPUT : DMA high and low bytes address values
DESCRIPTION  : Set the DMA buffers starting address
-----------------------------------------------------------------------------*/                                                  
void SetDmaAdd(Word Dmar)
{
	USBDMAR = Dmar/256;
	USBIDR = (Dmar%256) & 0xC0;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : CheckValidResume
INPUT/OUTPUT : None
DESCRIPTION  : Function called by INT_EndSusp ISR.
-----------------------------------------------------------------------------*/
void CheckValidResume(void)
{
	USBLibStatus &= ~USB_SUSPEND;	// Inform app code that suspend is over.
}

/******************** USB INTERRUPTS SERVICE ROUTINES ************************/

/*-----------------------------------------------------------------------------

ROUTINE NAME : INT_EndSusp
DESCRIPTION  : USB End of Suspend mode interrupt routine.
				Called when host sends a resume signal
-----------------------------------------------------------------------------*/ 
#ifdef COSMIC
@interrupt void INT_EndSusp(void)
#endif
#ifdef HIWARE
#pragma TRAP_PROC SAVE_REGS
void INT_EndSusp(void)
#endif
{
	CheckValidResume();		// Check if valid resume and Reset suspend flags.
	USBISTR = ~Int_Esusp;	// Clear pending bit    
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : INT_Usb
DESCRIPTION  : Comes from the USB cell.
-----------------------------------------------------------------------------*/ 
#ifdef COSMIC
@interrupt void INT_Usb(void)
#endif
#ifdef HIWARE
#pragma TRAP_PROC SAVE_REGS
void INT_Usb(void)
#endif
{// WARNING !! DO NOT CHANGE THE ORDER OF THE FOLLOWING STATEMENTS !!
	Byte	EndPointNumber, PacketID;

	if (USBISTR & Int_Reset)
	{
          UsbBusReset(); // USB Bus Reset Service routine.
          USBISTR = 0; // Clear USBISTR (USB Reset is the highest priority).
	}

	if (USBISTR & Int_Ctr)
	{	
		EndPointNumber = (USBIDR & MASK_EP)>>4;	// Get EndPoint number.
		PacketID = (USBPIDR & TP);				// Get Packet ID.

		if (EndPointNumber == 0)		// Test if Ctr IT occured on EP0
		{   // -> decode PacketID      
			if (PacketID == TP_SETUP)
			{
				UsbCtrFlag |= Int_Ctr_SETUP;	// Set Int_Ctr_SETUP flag
			}
			else
			if (PacketID == TP_IN) 
			{   
				if (USBTransferStatus & DATA_STAGE_IN)
				{	// Data Stage IN
					SetEP0RxStatus_IT(NAK); 	// set OUT direction to NACK
                    			UsbCtrFlag |= Int_Ctr_IN;       // Set Int_Ctr_IN flag
				}
				else
				{	// Data Stage OUT or No Data Stage -> Status IN
					if (USBTransferStatus & ADDRESS2SET)
					{
						SetAddress();		// Address actually set here
					}
					SetEP0TxStatus_IT(STALL);	//  Next IN will be STALLed.
					SetEP0RxStatus_IT(STALL);	//  Next OUT will be STALLed.
					//User_Status_In();		// Call-back function
				}					
			}
			else 
			{   // PacketID = TP_OUT
				if (USBTransferStatus & DATA_STAGE_OUT)
				{	// Data stage OUT
					SetEP0TxStatus_IT(NAK);      			// set IN direction to NACK
                    			UsbCtrFlag |= Int_Ctr_OUT;  		    // Set Int_Ctr_OUT flag
					OUT_DataNumberEP0 = USBIDR & MASK_CNT;	// Save Number of received data.
				}
				else
				{	// No data stage (not supposed to happen!) or DATA_STAGE_IN (->status stage OUT)
					SetEP0TxStatus_IT(STALL);	//  Next IN will be STALLed.
					//SetEP0RxStatus_IT(STALL);	//  Next OUT will be ACKed.
                			User_Status_Out();		// Call-back function                     
				}
			}
		}   // end of EP0 ISR
		USBISTR = ~Int_Ctr;					// Clear pending bit
	} // end of USB Correct Transfer (CTR) ISR

	if (USBISTR & Int_Susp)
	{                                
		USBCTLR = CtrlSusp;		// Set Suspend bit in USBCTLR to actually start logic suspend
		USBLibStatus |= USB_SUSPEND;	// Inform app code that suspend has started
		USBISTR = ~Int_Susp;			// Clear pending bit
	}
}

#ifdef HIWARE
#pragma CODE_SEG DEFAULT
#endif

/*** END OF FILE ***/