/******************** (C) COPYRIGHT 2006 STMicroelectronics ********************
* File Name          : device_RFID.h
* Author             : MPA Systems Lab
* Date First Issued  : 31/01/2006
* Description        : RFID Management for SR176 tag
********************************************************************************
* History:
*  31/01/2006 : V1.0
*******************************************************************************
 THE PRESENT SOFTWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS WITH
 CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
 AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT
 OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE CONTENT
 OF SUCH SOFTWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING INFORMATION
 CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
*******************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "71x_lib.h"


/*******************************************************************************************/
/*																						   */
/*					C function for using CRX14											   */
/*																						   */
/*******************************************************************************************/


/* Defines ------------------------------------------------------------------*/
#define	UCHAR	unsigned char
#define I2C_frequency 400000
 

/* Private typedef -----------------------------------------------------------*/
typedef enum
{
  SR176_POWEROFF	= 0x00,
  SR176_READY		= 0x01,
  SR176_ACTIVE		= 0x02,
  SR176_SELECTED	= 0x03,
  SR176_DESELECTED	= 0x04,
  SR176_DEACTIVTED	= 0x05
} SR176_State;

typedef enum
{
  CRX14_PR  = 0x00,
  CRX14_IOFR= 0x01,
  CRX14_AR  = 0x02,
  CRX14_SMR = 0x03
} CRX14_Register;

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/

/* Private function prototypes -----------------------------------------------*/






/*******************************************************************************
* Function Name  : Write_RFID
* Description    : Write CRX14 Registers
* Input          : 
*					addr 		- CRX14 register address										   
*					data2write	- buffer pointer (buffer contains data to write in RFID)		   
*					lenght		- number of byte to send										   
* Output         : None.
* Return         : Status.
*					1			- error  														   
*					0			- no error														   
*******************************************************************************/
extern u8  Write_RFID(CRX14_Register addr, u8 *data2write, u8 lenght);

/*******************************************************************************
* Function Name  : Read_RFID
* Description    : Read CRX14 Registers
* Input          : 
*					addr 		- CRX14 register address										   
*					data2read	- buffer pointer												   
*					lenght		- buffer lenght													   
* Output         : None.
* Return         : Status.
*					1			- error  														   
*					0			- no error														   
*******************************************************************************/
extern u8  Read_RFID(CRX14_Register addr, u8 *data2read, u8 *lenght);

/*******************************************************************************
* Function Name  : Polling
* Description    : Wait device until it is ready
* Input          : i2c address.
* Output         : None.
* Return         : Status.
******************************************************************************/
extern int Polling(int addr);							// Polling	waits device ready

/*******************************************************************************
* Function Name  : Write_CRX14_IOFR
* Description    : Write IO Frame Register CRX14
* Input          : data to send.
* Output         : None.
* Return         : Status.
******************************************************************************/
extern int Write_CRX14_IOFR(UCHAR *DATA);

/*******************************************************************************
* Function Name  : Read_CRX14_IOFR
* Description    : Read IO Frame Register CRX14
* Input          : None.
* Output         : DATA.
*
* Return         : Status.
******************************************************************************/
extern int Read_CRX14_IOFR(UCHAR *DATA);

/*******************************************************************************
* Function Name  : Write_CRX14_PR
* Description    : Write Parameter Register CRX14
* Input          : data to send.
* Output         : None.
*
* Return         : Status.
*					1 - No error
*					0 - error
******************************************************************************/
extern int Write_CRX14_PR(UCHAR *DATA);

/*******************************************************************************
* Function Name  : Read_CRX14_PR
* Description    : Read Parameter Register CRX14
* Input          : None.
* Output         : Data.
*
* Return         : Status.
******************************************************************************/
extern int Read_CRX14_PR(UCHAR *DATA);


/*******************************************************************************
* Function Name  : Write_Authenticate
* Description    : Write Authenticate Register CRX14
* Input          : addr ,DATA.
* Output         : None.
*
* Return         : Status.
******************************************************************************/
extern int Write_Authenticate(int addr ,UCHAR *DATA);	// Write Authenticate Register

/*******************************************************************************
* Function Name  : Write_Slot
* Description    : Write Slot Marker Register CRX14
* Input          : addr.
* Output         : None.
*
* Return         : Status.
******************************************************************************/
extern int Write_Slot(int addr);						// Write Slot Marker Register

/*******************************************************************************
* Function Name  : Read_Sign
* Description    : Read Sign Register CRX14
* Input          : addr.
* Output         : Data.
*
* Return         : Status.
******************************************************************************/
extern int Read_Sign(int addr ,UCHAR *DATA);			// Read Sign register

/*******************************************************************************
* Function Name  : Read_slot
* Description    : Read Slot Register CRX14
* Input          : addr.
* Output         : Data.
*
* Return         : Status.
*					1 - No error
*					0 - error
******************************************************************************/
extern int Read_slot(int addr ,UCHAR *DATA);			// Read Slot Marker Register

/*******************************************************************************
* Function Name  : Time_Read
* Description    : 
* Input          : None.
* Output         : None.
*
* Return         : Status.
******************************************************************************/
extern int Time_Read (void);							// Function to send Parameter to CRX14

/*******************************************************************************
* Function Name  : Time_Authenticate
* Description    : 
* Input          : None.
* Output         : None.
*
* Return         : Status.
******************************************************************************/
extern int Time_Authenticate (void);					// Function to send Parameter to CRX14

/*******************************************************************************
* Function Name  : Time_Write
* Description    : 
* Input          : None.
* Output         : None.
*
* Return         : Status.
******************************************************************************/
extern int Time_Write (void);							// Function to send Parameter to CRX14

/*******************************************************************************
* Function Name  : RF_OFF
* Description    : Cut off RF carrier
* Input          : None.
* Output         : None.
*
* Return         : Status.
******************************************************************************/
extern int RF_OFF (void);								// Function to send Parameter to CRX14

/*******************************************************************************
* Function Name  : RF_ON
* Description    : Switch on RF carrier
* Input          : None.
* Output         : None.
*
* Return         : Status.
******************************************************************************/
extern int RF_ON (void);								// Function to send Parameter to CRX14

/*******************************************************************************
* Function Name  : Send_Receive_Data
* Description    : 
* 					This function send Trame (TX) to CRX14 using the Write_IO function and				   
*					Receive Trame from CRX14 using the Read_IO_data function.						   
* Input          :  TX
* Output         :  RX
*
* Return         : Status.
******************************************************************************/
extern int Send_Receive_Data(UCHAR *TX, UCHAR *RX, int Nb_RX);	//send & receive Trame to CRX14

/*******************************************************************************
* Function Name  : Send_Receive_Authenticate
* Description    : 
* 					This function send Trame (TX) to CRX14 using the Write_Authenticate function and				   
*					Receive Trame from CRX14 using the Read_Sign function.						   
* Input          :  TX
* Output         :  RX
*
* Return         : Status.
******************************************************************************/
extern int Send_Receive_Authenticate(UCHAR *TX, UCHAR *RX, int Nb_RX);

/*******************************************************************************
* Function Name  : Send_Receive_Inventory
* Description    : 
* 					This function start an automated anti-collision sequence
*					If there's selected Tag in the field this function sends Anti-collision sequence   
*					and returns the status and potential CHIP-ID in each time slots.				   
* Input          : None.
* Output         : RX
*
* Return         : Status.
* Note			 :  IMPORTANT - FUNCTION NOT IMPLEMENTED
******************************************************************************/
extern int Send_Receive_Inventory(UCHAR *RX, int Nb_RX);


/*******************************************************************************************/
/*																						   */
/*					C function for using SR176											   */
/*																						   */
/*******************************************************************************************/

/*******************************************************************************
* Function Name  : Initiate
* Description    : Initiate Tag
*					If there's Tag in the field this function returns the Chip ID					   
*					Trame transmit to the CRX14														   
* Input          : None.
* Output         : RX.
* Return         : Status.
*******************************************************************************/
extern int Initiate (UCHAR *RX);								// FUNCTION TO INITIATE TAG

/*******************************************************************************
* Function Name  : Select
* Description    : Select Tag
*					If there's Tag in the field this function returns the Chip ID					   
*					Trame transmit to the CRX14														   
* Input          : CHIP_ID.
* Output         : RX.
* Return         : Status.
*******************************************************************************/
extern int Select (UCHAR *CHIP_ID,UCHAR *RX);					// FUNCTION TO SELECT TAG

/*******************************************************************************
* Function Name  : ReadBlock
* Description    : Read a block Tag
*					If there's selected Tag in the field this function returns the content			   
*					of the block specified by the adress.											   
* Input          : 
*					Addr: adress of the block														   
* Output         : 
*					RX[1]... RX[n]: Content of the block											   
* Return         : Status.
******************************************************************************/
extern int ReadBlock (UCHAR Addr, UCHAR *RX);					// FUNCTION TO READ BLOCK TAG

/*******************************************************************************
* Function Name  : WriteBlock
* Description    : Write a block Tag
*					If there's selected Tag in the field this function writes array of yy data		   
*					in Addr block and return no answers.											   
* Input          : 
*					Addr: adress of the block														   
* 					DATA_WR: array of data to write in Addr block									   
* Output         : 
*					RX[1]... RX[n]: Content of the block											   
* Return         : Status.
******************************************************************************/
extern int WriteBlock (UCHAR Addr, UCHAR *DATA_WR, UCHAR *RX);	// FUNCTION TO WRITE  BLOCK TAG

/*******************************************************************************
* Function Name  : Completion
* Description    : Send completion command to the tag
*					If there's selected Tag in the field this function send completion command		   
* Input          : 
* Output         : RX
* Return         : Status.
******************************************************************************/
extern int Completion (UCHAR *RX);								// FUNCTION TO SEND COMPLETION COMMAND

/*******************************************************************************
* Function Name  : Get_Protect
* Description    : Send get protect command to the tag
*					If there's Tag in the field this function send get protect command				   
* Input          : None.
* Output         : RX
* Return         : Status.
******************************************************************************/
extern int Get_Protect (UCHAR *RX);							// FUNCTION TO SEND GET_PROTECT COMMAND

/*******************************************************************************
* Function Name  : Protect_Block
* Description    : Send protect block command to the tag
*					If there's Tag in the field this function send get protect command				   
* Input          : 
*					LOCK_REG: Protection register													   
* Output         : 
*					RX[0]=0			Number of receive byte											   
* Return         : Status.
******************************************************************************/
extern int Protect_Block (UCHAR *Lock_Reg, UCHAR *RX); 		// FUNCTION TO SEND PROTECT_BLOCK COMMAND


/*******************************************************************************
* Function Name  : init_CRX14_I2C
* Description    : Configure and init I2C in order to communicates with CRX14
* Input          : None.
* Output         : None.
* Return         : Status.
*******************************************************************************/
extern void init_CRX14_I2C(void);


/*******************************************************************************
* Function Name  : RFID_Handler
* Description    : Manage SR176 RFID tag
* Input          : (RFID_Action, TX_frame, TX_lenght, RX_frame, RX_lenght, RX_offset, RFID_UID, RFID_timeout)
* Output         : (RX_frame, RFID_UID)
* Return         : Status.
*******************************************************************************/
extern u8 RFID_Handler(u8 RFID_Action, u8 *TX_frame, u8 TX_lenght, u8 TX_offset, u8 *RX_frame, u8 RX_lenght, u8 RX_offset, u32 *RFID_UID, u8 RFID_timeout);

/*******************************************************************************
* Function Name  : init_RFID
* Description    : Initialize RFID communication
* Input          : 
*					RFID_CHIPID	- tag ID														   
*					SR176State	- tag State														   
* Output         : None.
* Return         : Status.
*					1			- error  														   
*					0			- no error														   
*******************************************************************************/
extern u8 init_RFID(UCHAR *RFID_CHIPID, SR176_State *SR176State);

/*******************************************************************************
* Function Name  : GetSR176_64UID
* Description    : Get SR176 64 bit unique ID
* Input          : 
*					RFID_UID	- buffer pointer to SR176 unique id								   
* Output         : None.
* Return         : Status.
*					1			- error  														   
*					0			- no error														   
*******************************************************************************/
extern u8 GetSR176_64UID(u32 *RFID_UID);

/*******************************************************************************
* Function Name  : WriteSR176_EEPROM
* Description    : Write SR176 RFID EEPROM
* Input          : 
*					TX_frame	- buffer pointer to TX frame to write in SR176 EEPROM			   
*					EEPROMoffset- EEPROM offset where to write									   
*					TX_frame_lenght	- buffer lenght												   
* Output         : None.
* Return         : Status.
*					1			- error  														   
*					0			- no error														   
*******************************************************************************/
extern u8 WriteSR176_EEPROM(u8 *TX_frame, u8 EEPROMoffset,u8 TX_frame_lenght);

/*******************************************************************************
* Function Name  : ReadSR176_EEPROM
* Description    : Read SR176 RFID EEPROM
* Input          : 
*					RFID_Data	- buffer pointer to RX frame where to read in SR176 EEPROM		   
*					EEPROMoffset- EEPROM offset where to write									   
*					TX_frame_lenght	- buffer lenght												   
* Output         : None.
* Return         : Status.
*					1			- error  														   
*					0			- no error														   
*******************************************************************************/
extern u8 ReadSR176_EEPROM(u8 *RFID_Data, u8 EEPROMoffset,u8 RX_frame_lenght);


/******************* (C) COPYRIGHT 2006 STMicroelectronics *****END OF FILE****/





