;---------------------------------------------------------------------
; This software module is a Radix-4 FFT implementation for
; ARM9E validated in IAR/STR91x environment
; and the module is presented "as is with no warranty".
;---------------------------------------------------------------------
;
; DECLARATIONS, DATA, ALIAS, CONSTANTS, MACROS
;

BUTFLY4  MACRO pIN,offset,pOUT,s,pK
        LDR     R8, [pIN],-offset
        LDR     R9, [pK], #4
        LDR     R6, [pIN],-offset
        LDR     R7, [pK], #4
        LDR     R4, [pIN],-offset
        LDR     R5, [pK], #4
        ;;CXMUL Dr,Di,R8,R9,R12
        SMULBT  R12, R8, R9        ; sYr*sKi
        SMULBB  Dr, R8, R9         ; sYr*sKr
        SMULTB  Di, R8, R9         ; sYi*sYr
        SMLATT  Dr, R8, R9, Dr     ; lYYr= sYr*sKr+sYi*sKi
        SUB     Di, Di, R12        ; lYYi= sYi*sKr-sYr*sKi
        ;;CXMUL Cr,Ci,R6,R7,R12
        SMULBT  R12, R6, R7
        SMULBB  Cr, R6, R7
        SMULTB  Ci, R6, R7
        SMLATT  Cr, R6, R7, Cr     ; lYYr= sYr*sKr+sYi*sKi
        SUB     Ci, Ci, R12        ; lYYi= sYi*sKr-sYr*sKi
        ;;CXMUL Br,Bi,R4,R5,R12
        SMULBT  R12, R4, R5
        SMULBB  Br, R4, R5
        SMULTB  Bi, R4, R5
        SMLATT  Br, R4, R5, Br     ; lYYr= sYr*sKr+sYi*sKi
        SUB     Bi, Bi, R12        ; lYYi= sYi*sKr-sYr*sKi
        LDRSH   Ai, [pIN, #2]
        LDRSH   Ar, [pIN]
        ; (C,D) = (C+D, C-D)
        ADD     Cr, Cr, Dr
        ADD     Ci, Ci, Di
        SUB     Dr, Cr, Dr, LSL#1
        SUB     Di, Ci, Di, LSL#1
        ; (A,B) = (A+(B>>s), A-(B>>s))/4
        MOV     Ar, Ar, ASR#2
        MOV     Ai, Ai, ASR#2
        ADD     Ar, Ar, Br, ASR#(2+s)
        ADD     Ai, Ai, Bi, ASR#(2+s)
        SUB     Br, Ar, Br, ASR#(1+s)
        SUB     Bi, Ai, Bi, ASR#(1+s)
        ; (A,C) = (A+(C>>s)/4, A-(C>>s)/4)
        ADD     Ar, Ar, Cr, ASR#(2+s)
        ADD     Ai, Ai, Ci, ASR#(2+s)
        SUB     Cr, Ar, Cr, ASR#(1+s)
        SUB     Ci, Ai, Ci, ASR#(1+s)
        ; (B,D) = (B-i*(D>>s)/4, B+i*(D>>s)/4)
        ADD     Br, Br, Di, ASR#(2+s)
        SUB     Bi, Bi, Dr, ASR#(2+s)
        SUB     Di, Br, Di, ASR#(1+s)
        ADD     Dr, Bi, Dr, ASR#(1+s)
        ;
        STRH    Ai, [pOUT, #2]
        STRH    Ar, [pOUT], offset
        STRH    Bi, [pOUT, #2]
        STRH    Br, [pOUT], offset
        STRH    Ci, [pOUT, #2]
        STRH    Cr, [pOUT], offset
        STRH    Dr, [pOUT, #2]  ; inversion here
        STRH    Di, [pOUT], #4
        ENDM

BUTFLY4ZERO  MACRO pIN,offset,pOUT
        LDRSH   Ai, [pIN, #2]
        LDRSH   Ar, [pIN],offset
        LDRSH   Ci, [pIN, #2]
        LDRSH   Cr, [pIN],offset
        LDRSH   Bi, [pIN, #2]
        LDRSH   Br, [pIN],offset
        LDRSH   Di, [pIN, #2]
        LDRSH   Dr, [pIN],offset
        ; (C,D) = (C+D, C-D)
        ADD     Cr, Cr, Dr
        ADD     Ci, Ci, Di
        SUB     Dr, Cr, Dr, LSL#1  ; trick
        SUB     Di, Ci, Di, LSL#1  ;trick
        ; (A,B) = (A+B)/4, (A-B)/4
        MOV     Ar, Ar, ASR#2
        MOV     Ai, Ai, ASR#2
        ADD     Ar, Ar, Br, ASR#2
        ADD     Ai, Ai, Bi, ASR#2
        SUB     Br, Ar, Br, ASR#1
        SUB     Bi, Ai, Bi, ASR#1
        ; (A,C) = (A+C)/4, (A-C)/4
        ADD     Ar, Ar, Cr, ASR#2
        ADD     Ai, Ai, Ci, ASR#2
        SUB     Cr, Ar, Cr, ASR#1
        SUB     Ci, Ai, Ci, ASR#1
        ; (B,D) = (B-i*D)/4, (B+i*D)/4
        ADD     Br, Br, Di, ASR#2
        SUB     Bi, Bi, Dr, ASR#2
        SUB     Di, Br, Di, ASR#1
        ADD     Dr, Bi, Dr, ASR#1
        ;
        STRH    Ai, [pOUT, #2]
        STRH    Ar, [pOUT], #4
        STRH    Bi, [pOUT, #2]
        STRH    Br, [pOUT], #4
        STRH    Ci, [pOUT, #2]
        STRH    Cr, [pOUT], #4
        STRH    Dr, [pOUT, #2]  ; inversion here
        STRH    Di, [pOUT], #4
        ENDM

BUTFLY4K  MACRO pIN,offset,pOUT,s,pK
        LDR     D, [pIN],#-offset
        LDR     KD, [pK], #4
        LDR     C, [pIN],#-offset
        LDR     KC, [pK], #4
        LDR     B, [pIN],#-offset
        LDR     KB, [pK], #4
        ;;CXMUL R10,R11,R8,R9
        SMULBT  tmp, D, KD
        SMULBB  Dr, D, KD         ;
        SMULTB  Di, D, KD
        SMLATT  Dr, D, KD, Dr     ; lYYr= sYr*sKr+sYi*sKi
        SUB     Di, Di, tmp       ; lYYi= sYi*sKr-sYr*sKi
        ;;CXMUL R8,R9,R6,R7
        SMULBT  tmp, C, KC
        SMULBB  Cr, C, KC
        SMULTB  Ci, C, KC
        SMLATT  Cr, C, KC, Cr      ; lYYr= sYr*sKr+sYi*sKi
        SUB     Ci, Ci, tmp        ; lYYi= sYi*sKr-sYr*sKi
        ;;CXMUL R6,R7,R4,R5
        SMULBT  tmp, B, KB
        SMULBB  Br, B, KB
        SMULTB  Bi, B, KB
        SMLATT  Br, B, KB, Br      ; lYYr= sYr*sKr+sYi*sKi
        SUB     Bi, Bi, tmp        ; lYYi= sYi*sKr-sYr*sKi
        LDRSH   Ai, [pIN, #2]
        LDRSH   Ar, [pIN]
        ; (C,D) = (C+D, C-D)
        ADD     Cr, Cr, Dr
        ADD     Ci, Ci, Di
        SUB     Dr, Cr, Dr, LSL#1
        SUB     Di, Ci, Di, LSL#1
        ; (A,B) = (A+(B>>s), A-(B>>s))/4
        MOV     Ar, Ar, ASR#2
        MOV     Ai, Ai, ASR#2
        ADD     Ar, Ar, Br, ASR#(2+s)
        ADD     Ai, Ai, Bi, ASR#(2+s)
        SUB     Br, Ar, Br, ASR#(1+s)
        SUB     Bi, Ai, Bi, ASR#(1+s)
        ; (A,C) = (A+(C>>s)/4, A-(C>>s)/4)
        ADD     Ar, Ar, Cr, ASR#(2+s)
        ADD     Ai, Ai, Ci, ASR#(2+s)
        SUB     Cr, Ar, Cr, ASR#(1+s)
        SUB     Ci, Ai, Ci, ASR#(1+s)
        ; (B,D) = (B-i*(D>>s)/4, B+i*(D>>s)/4)
        ADD     Br, Br, Di, ASR#(2+s)
        SUB     Bi, Bi, Dr, ASR#(2+s)
        SUB     Di, Br, Di, ASR#(1+s)
        ADD     Dr, Bi, Dr, ASR#(1+s)
        ;
        STRH    Ai, [pOUT, #2]
        STRH    Ar, [pOUT], #offset
        STRH    Bi, [pOUT, #2]
        STRH    Br, [pOUT], #offset
        STRH    Ci, [pOUT, #2]
        STRH    Cr, [pOUT], #offset
        STRH    Dr, [pOUT, #2]  ; inversion here
        STRH    Di, [pOUT], #4
        ENDM


;################# FFT CODE SECTION #################
;---- pass 01 exclusive -----------------------------

#define pssOUTg  R0
#define pssINg   R1
#define gNbin    R2
#define gmaskbitrev R3
#define gcntrbitrev R12
#define pssIN2g  R14
;---------------------
#define pssKg   R0
#define pssXg   R1
#define gbutternbr   R2
#define gindex       R3
#define Ar     R4
#define Ai     R5
#define B      R4
#define KB     R5
#define Br     R6
#define Bi     R7
#define C      R6
#define KC     R7
#define Cr     R8
#define Ci     R9
#define D      R8
#define KD     R9
#define Dr     R10
#define Di     R11
#define tmp    R12
#define tmp2   R14

;---------------------

        PUBLIC cr4fft1k16_arm9e
        RSEG ICODE:CODE(4)
 	CODE32

;---------------------------------------------------------------------
;void cr4fft1k16_arm9e(void *pssOUT, void *pssIN, int nBin)

cr4fft1k16_arm9e:
;CALL
; R0 = pssOUTg
; R1 = pssINg
; R2 = gNbin
;---------------------------------------------------------------------
      STMFD   SP!, {R4-R11,LR}
;-pass01---------------------------
;USED RESOURCES          init value
; R0 = pssOUTg           passed by caller
; R1 = pssINg            passed by caller
; R2 = gNbin             passed by caller
; R3 =  gmaskbitrev      0x7FFFFFFF
; R4-R11 = A,B,C,D
; R12 = gcntrbitrev      0x0
; R14 = pssIN2g          f(pssIN)
;---------------------------------
        MOV     gcntrbitrev, #0
        MVN     gmaskbitrev, #0x80000000                        ; R=0x7FFFFFFF
preloop
        ADDS   pssIN2g, pssINg, gcntrbitrev, LSL#2              ; pssIN2= pssIN[cntrbr] and clear carry
        BUTFLY4ZERO pssIN2g,gNbin,pssOUTg
        ; bit reversed increment modulo (Nbin/4)
        RSC    gcntrbitrev, gcntrbitrev, gNbin, LSR #2          ; cntrbr = (Nbin/4)- cntrbr - 1
        CLZ    tmp2, gcntrbitrev                                ; find leading 1
        EORS   gcntrbitrev, gcntrbitrev, gmaskbitrev, ASR tmp2  ; toggle bits below leading 1
        BNE    preloop
;
;-pass23,pass45-------------------
;USED RESOURCES        init value
; R0 = pssKg           = pssKg[0] = Table FFT
; R1 = pssXg           = pssOUT - Nbin
; R2 = gbutternbr      = f(Nbin)
; R3 = gindex          =16
; R4-R11 = A,B,C,D
; R4 = Ar ; R5 = Ai
; R6 = Br ; R7 = Bi
; R8 = Cr ; R9 = Ci
; R10 =Dr ; R11 = Di
; R12 = gtmp
; R14 = gtmp2
;-----------------------
        SUB     pssXg, pssOUTg, gNbin, LSL #2
        MOV     gindex, #16
        MOVS    gbutternbr, gNbin, LSR#4
        ADRL    pssKg, Table_FFT
;-----------------------
passloop
        STMFD   SP!, {pssXg, gbutternbr}
        ADD     tmp, gindex, gindex, LSL#1
        ADD     pssXg, pssXg, tmp
        SUB     gbutternbr,gbutternbr, #1<<16
;-----------------------
grouploop
        ADD     gbutternbr, gbutternbr, gindex, LSL#(16-2)
;-----------------------
butterloop
        BUTFLY4 pssXg,gindex,pssXg,15,pssKg
        SUBS    gbutternbr, gbutternbr, #1<<16
        BGE     butterloop
;-----------------------
        ADD     tmp, gindex, gindex, LSL#1
        ADD     pssXg, pssXg, tmp
        SUB     gbutternbr, gbutternbr, #1
        MOVS    tmp2, gbutternbr, LSL#16        ; used for status only
        SUBNE   pssKg, pssKg, tmp               ; pssK  += index*2.5
        BNE     grouploop
;-----------------------
        LDMFD   SP!, {pssXg, gbutternbr}
        MOV     gindex, gindex, LSL#2            ; index *= radix
        MOVS    gbutternbr, gbutternbr, LSR#2    ; loop nbr /= radix
        BNE     passloop
;-----------------------
epilog
        LDMFD   sp!, {R4-R11, PC}


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
Table_FFT
        ; N=16
        DC16 0x7fff,0x0000, 0x7fff,0x0000, 0x7fff,0x0000, 0x30fc,0x7642, 0x7642,0x30fc, 0x5a82,0x5a82
        DC16 0xa57e,0x5a82, 0x5a82,0x5a82, 0x0000,0x7fff, 0x89be,0xcf04, 0x30fc,0x7642, 0xa57e,0x5a82
        ; N=64
        DC16 0x7fff,0x0000, 0x7fff,0x0000, 0x7fff,0x0000, 0x7a7d,0x2528, 0x7f62,0x0c8c, 0x7d8a,0x18f9
        DC16 0x6a6e,0x471d, 0x7d8a,0x18f9, 0x7642,0x30fc, 0x5134,0x62f2, 0x7a7d,0x2528, 0x6a6e,0x471d
        DC16 0x30fc,0x7642, 0x7642,0x30fc, 0x5a82,0x5a82, 0x0c8c,0x7f62, 0x70e3,0x3c57, 0x471d,0x6a6e
        DC16 0xe707,0x7d8a, 0x6a6e,0x471d, 0x30fc,0x7642, 0xc3a9,0x70e3, 0x62f2,0x5134, 0x18f9,0x7d8a
        DC16 0xa57e,0x5a82, 0x5a82,0x5a82, 0x0000,0x7fff, 0x8f1d,0x3c57, 0x5134,0x62f2, 0xe707,0x7d8a
        DC16 0x8276,0x18f9, 0x471d,0x6a6e, 0xcf04,0x7642, 0x809e,0xf374, 0x3c57,0x70e3, 0xb8e3,0x6a6e
        DC16 0x89be,0xcf04, 0x30fc,0x7642, 0xa57e,0x5a82, 0x9d0e,0xaecc, 0x2528,0x7a7d, 0x9592,0x471d
        DC16 0xb8e3,0x9592, 0x18f9,0x7d8a, 0x89be,0x30fc, 0xdad8,0x8583, 0x0c8c,0x7f62, 0x8276,0x18f9
        ; N=256
        DC16 0x7fff,0x0000, 0x7fff,0x0000, 0x7fff,0x0000, 0x7fa7,0x096b, 0x7ff6,0x0324, 0x7fd9,0x0648
        DC16 0x7e9d,0x12c8, 0x7fd9,0x0648, 0x7f62,0x0c8c, 0x7ce4,0x1c0c, 0x7fa7,0x096b, 0x7e9d,0x12c8
        DC16 0x7a7d,0x2528, 0x7f62,0x0c8c, 0x7d8a,0x18f9, 0x776c,0x2e11, 0x7f0a,0x0fab, 0x7c2a,0x1f1a
        DC16 0x73b6,0x36ba, 0x7e9d,0x12c8, 0x7a7d,0x2528, 0x6f5f,0x3f17, 0x7e1e,0x15e2, 0x7885,0x2b1f
        DC16 0x6a6e,0x471d, 0x7d8a,0x18f9, 0x7642,0x30fc, 0x64e9,0x4ec0, 0x7ce4,0x1c0c, 0x73b6,0x36ba
        DC16 0x5ed7,0x55f6, 0x7c2a,0x1f1a, 0x70e3,0x3c57, 0x5843,0x5cb4, 0x7b5d,0x2224, 0x6dca,0x41ce
        DC16 0x5134,0x62f2, 0x7a7d,0x2528, 0x6a6e,0x471d, 0x49b4,0x68a7, 0x798a,0x2827, 0x66d0,0x4c40
        DC16 0x41ce,0x6dca, 0x7885,0x2b1f, 0x62f2,0x5134, 0x398d,0x7255, 0x776c,0x2e11, 0x5ed7,0x55f6
        DC16 0x30fc,0x7642, 0x7642,0x30fc, 0x5a82,0x5a82, 0x2827,0x798a, 0x7505,0x33df, 0x55f6,0x5ed7
        DC16 0x1f1a,0x7c2a, 0x73b6,0x36ba, 0x5134,0x62f2, 0x15e2,0x7e1e, 0x7255,0x398d, 0x4c40,0x66d0
        DC16 0x0c8c,0x7f62, 0x70e3,0x3c57, 0x471d,0x6a6e, 0x0324,0x7ff6, 0x6f5f,0x3f17, 0x41ce,0x6dca
        DC16 0xf9b8,0x7fd9, 0x6dca,0x41ce, 0x3c57,0x70e3, 0xf055,0x7f0a, 0x6c24,0x447b, 0x36ba,0x73b6
        DC16 0xe707,0x7d8a, 0x6a6e,0x471d, 0x30fc,0x7642, 0xdddc,0x7b5d, 0x68a7,0x49b4, 0x2b1f,0x7885
        DC16 0xd4e1,0x7885, 0x66d0,0x4c40, 0x2528,0x7a7d, 0xcc21,0x7505, 0x64e9,0x4ec0, 0x1f1a,0x7c2a
        DC16 0xc3a9,0x70e3, 0x62f2,0x5134, 0x18f9,0x7d8a, 0xbb85,0x6c24, 0x60ec,0x539b, 0x12c8,0x7e9d
        DC16 0xb3c0,0x66d0, 0x5ed7,0x55f6, 0x0c8c,0x7f62, 0xac65,0x60ec, 0x5cb4,0x5843, 0x0648,0x7fd9
        DC16 0xa57e,0x5a82, 0x5a82,0x5a82, 0x0000,0x7fff, 0x9f14,0x539b, 0x5843,0x5cb4, 0xf9b8,0x7fd9
        DC16 0x9930,0x4c40, 0x55f6,0x5ed7, 0xf374,0x7f62, 0x93dc,0x447b, 0x539b,0x60ec, 0xed38,0x7e9d
        DC16 0x8f1d,0x3c57, 0x5134,0x62f2, 0xe707,0x7d8a, 0x8afb,0x33df, 0x4ec0,0x64e9, 0xe0e6,0x7c2a
        DC16 0x877b,0x2b1f, 0x4c40,0x66d0, 0xdad8,0x7a7d, 0x84a3,0x2224, 0x49b4,0x68a7, 0xd4e1,0x7885
        DC16 0x8276,0x18f9, 0x471d,0x6a6e, 0xcf04,0x7642, 0x80f6,0x0fab, 0x447b,0x6c24, 0xc946,0x73b6
        DC16 0x8027,0x0648, 0x41ce,0x6dca, 0xc3a9,0x70e3, 0x800a,0xfcdc, 0x3f17,0x6f5f, 0xbe32,0x6dca
        DC16 0x809e,0xf374, 0x3c57,0x70e3, 0xb8e3,0x6a6e, 0x81e2,0xea1e, 0x398d,0x7255, 0xb3c0,0x66d0
        DC16 0x83d6,0xe0e6, 0x36ba,0x73b6, 0xaecc,0x62f2, 0x8676,0xd7d9, 0x33df,0x7505, 0xaa0a,0x5ed7
        DC16 0x89be,0xcf04, 0x30fc,0x7642, 0xa57e,0x5a82, 0x8dab,0xc673, 0x2e11,0x776c, 0xa129,0x55f6
        DC16 0x9236,0xbe32, 0x2b1f,0x7885, 0x9d0e,0x5134, 0x9759,0xb64c, 0x2827,0x798a, 0x9930,0x4c40
        DC16 0x9d0e,0xaecc, 0x2528,0x7a7d, 0x9592,0x471d, 0xa34c,0xa7bd, 0x2224,0x7b5d, 0x9236,0x41ce
        DC16 0xaa0a,0xa129, 0x1f1a,0x7c2a, 0x8f1d,0x3c57, 0xb140,0x9b17, 0x1c0c,0x7ce4, 0x8c4a,0x36ba
        DC16 0xb8e3,0x9592, 0x18f9,0x7d8a, 0x89be,0x30fc, 0xc0e9,0x90a1, 0x15e2,0x7e1e, 0x877b,0x2b1f
        DC16 0xc946,0x8c4a, 0x12c8,0x7e9d, 0x8583,0x2528, 0xd1ef,0x8894, 0x0fab,0x7f0a, 0x83d6,0x1f1a
        DC16 0xdad8,0x8583, 0x0c8c,0x7f62, 0x8276,0x18f9, 0xe3f4,0x831c, 0x096b,0x7fa7, 0x8163,0x12c8
        DC16 0xed38,0x8163, 0x0648,0x7fd9, 0x809e,0x0c8c, 0xf695,0x8059, 0x0324,0x7ff6, 0x8027,0x0648
        ;1024
        DC16 0x7fff,0x0000, 0x7fff,0x0000, 0x7fff,0x0000, 0x7ffa,0x025b, 0x7fff,0x00c9, 0x7ffe,0x0192
        DC16 0x7fea,0x04b6, 0x7ffe,0x0192, 0x7ff6,0x0324, 0x7fce,0x0711, 0x7ffa,0x025b, 0x7fea,0x04b6
        DC16 0x7fa7,0x096b, 0x7ff6,0x0324, 0x7fd9,0x0648, 0x7f75,0x0bc4, 0x7ff1,0x03ed, 0x7fc2,0x07d9
        DC16 0x7f38,0x0e1c, 0x7fea,0x04b6, 0x7fa7,0x096b, 0x7ef0,0x1073, 0x7fe2,0x057f, 0x7f87,0x0afb
        DC16 0x7e9d,0x12c8, 0x7fd9,0x0648, 0x7f62,0x0c8c, 0x7e3f,0x151c, 0x7fce,0x0711, 0x7f38,0x0e1c
        DC16 0x7dd6,0x176e, 0x7fc2,0x07d9, 0x7f0a,0x0fab, 0x7d63,0x19be, 0x7fb5,0x08a2, 0x7ed6,0x113a
        DC16 0x7ce4,0x1c0c, 0x7fa7,0x096b, 0x7e9d,0x12c8, 0x7c5a,0x1e57, 0x7f98,0x0a33, 0x7e60,0x1455
        DC16 0x7bc6,0x209f, 0x7f87,0x0afb, 0x7e1e,0x15e2, 0x7b27,0x22e5, 0x7f75,0x0bc4, 0x7dd6,0x176e
        DC16 0x7a7d,0x2528, 0x7f62,0x0c8c, 0x7d8a,0x18f9, 0x79c9,0x2768, 0x7f4e,0x0d54, 0x7d3a,0x1a83
        DC16 0x790a,0x29a4, 0x7f38,0x0e1c, 0x7ce4,0x1c0c, 0x7840,0x2bdc, 0x7f22,0x0ee4, 0x7c89,0x1d93
        DC16 0x776c,0x2e11, 0x7f0a,0x0fab, 0x7c2a,0x1f1a, 0x768e,0x3042, 0x7ef0,0x1073, 0x7bc6,0x209f
        DC16 0x75a6,0x326e, 0x7ed6,0x113a, 0x7b5d,0x2224
        DC16 0x74b3,0x3497, 0x7eba,0x1201, 0x7aef,0x23a7
        DC16 0x73b6,0x36ba, 0x7e9d,0x12c8, 0x7a7d,0x2528
        DC16 0x72af,0x38d9, 0x7e7f,0x138f, 0x7a06,0x26a8
        DC16 0x719e,0x3af3, 0x7e60,0x1455, 0x798a,0x2827
        DC16 0x7083,0x3d08, 0x7e3f,0x151c, 0x790a,0x29a4
        DC16 0x6f5f,0x3f17, 0x7e1e,0x15e2, 0x7885,0x2b1f
        DC16 0x6e31,0x4121, 0x7dfb,0x16a8, 0x77fb,0x2c99
        DC16 0x6cf9,0x4326, 0x7dd6,0x176e, 0x776c,0x2e11
        DC16 0x6bb8,0x4524, 0x7db1,0x1833, 0x76d9,0x2f87
        DC16 0x6a6e,0x471d, 0x7d8a,0x18f9, 0x7642,0x30fc
        DC16 0x691a,0x490f, 0x7d63,0x19be, 0x75a6,0x326e
        DC16 0x67bd,0x4afb, 0x7d3a,0x1a83, 0x7505,0x33df
        DC16 0x6657,0x4ce1, 0x7d0f,0x1b47, 0x7460,0x354e
        DC16 0x64e9,0x4ec0, 0x7ce4,0x1c0c, 0x73b6,0x36ba
        DC16 0x6371,0x5098, 0x7cb7,0x1cd0, 0x7308,0x3825
        DC16 0x61f1,0x5269, 0x7c89,0x1d93, 0x7255,0x398d
        DC16 0x6068,0x5433, 0x7c5a,0x1e57, 0x719e,0x3af3
        DC16 0x5ed7,0x55f6, 0x7c2a,0x1f1a, 0x70e3,0x3c57
        DC16 0x5d3e,0x57b1, 0x7bf9,0x1fdd, 0x7023,0x3db8
        DC16 0x5b9d,0x5964, 0x7bc6,0x209f, 0x6f5f,0x3f17
        DC16 0x59f4,0x5b10, 0x7b92,0x2162, 0x6e97,0x4074
        DC16 0x5843,0x5cb4, 0x7b5d,0x2224, 0x6dca,0x41ce
        DC16 0x568a,0x5e50, 0x7b27,0x22e5, 0x6cf9,0x4326
        DC16 0x54ca,0x5fe4, 0x7aef,0x23a7, 0x6c24,0x447b
        DC16 0x5303,0x616f, 0x7ab7,0x2467, 0x6b4b,0x45cd
        DC16 0x5134,0x62f2, 0x7a7d,0x2528, 0x6a6e,0x471d
        DC16 0x4f5e,0x646c, 0x7a42,0x25e8, 0x698c,0x486a
        DC16 0x4d81,0x65de, 0x7a06,0x26a8, 0x68a7,0x49b4
        DC16 0x4b9e,0x6747, 0x79c9,0x2768, 0x67bd,0x4afb
        DC16 0x49b4,0x68a7, 0x798a,0x2827, 0x66d0,0x4c40
        DC16 0x47c4,0x69fd, 0x794a,0x28e5, 0x65de,0x4d81
        DC16 0x45cd,0x6b4b, 0x790a,0x29a4, 0x64e9,0x4ec0
        DC16 0x43d1,0x6c8f, 0x78c8,0x2a62, 0x63ef,0x4ffb
        DC16 0x41ce,0x6dca, 0x7885,0x2b1f, 0x62f2,0x5134
        DC16 0x3fc6,0x6efb, 0x7840,0x2bdc, 0x61f1,0x5269
        DC16 0x3db8,0x7023, 0x77fb,0x2c99, 0x60ec,0x539b
        DC16 0x3ba5,0x7141, 0x77b4,0x2d55, 0x5fe4,0x54ca
        DC16 0x398d,0x7255, 0x776c,0x2e11, 0x5ed7,0x55f6
        DC16 0x3770,0x735f, 0x7723,0x2ecc, 0x5dc8,0x571e
        DC16 0x354e,0x7460, 0x76d9,0x2f87, 0x5cb4,0x5843
        DC16 0x3327,0x7556, 0x768e,0x3042, 0x5b9d,0x5964
        DC16 0x30fc,0x7642, 0x7642,0x30fc, 0x5a82,0x5a82
        DC16 0x2ecc,0x7723, 0x75f4,0x31b5, 0x5964,0x5b9d
        DC16 0x2c99,0x77fb, 0x75a6,0x326e, 0x5843,0x5cb4
        DC16 0x2a62,0x78c8, 0x7556,0x3327, 0x571e,0x5dc8
        DC16 0x2827,0x798a, 0x7505,0x33df, 0x55f6,0x5ed7
        DC16 0x25e8,0x7a42, 0x74b3,0x3497, 0x54ca,0x5fe4
        DC16 0x23a7,0x7aef, 0x7460,0x354e, 0x539b,0x60ec
        DC16 0x2162,0x7b92, 0x740b,0x3604, 0x5269,0x61f1
        DC16 0x1f1a,0x7c2a, 0x73b6,0x36ba, 0x5134,0x62f2
        DC16 0x1cd0,0x7cb7, 0x735f,0x3770, 0x4ffb,0x63ef
        DC16 0x1a83,0x7d3a, 0x7308,0x3825, 0x4ec0,0x64e9
        DC16 0x1833,0x7db1, 0x72af,0x38d9, 0x4d81,0x65de
        DC16 0x15e2,0x7e1e, 0x7255,0x398d, 0x4c40,0x66d0
        DC16 0x138f,0x7e7f, 0x71fa,0x3a40, 0x4afb,0x67bd
        DC16 0x113a,0x7ed6, 0x719e,0x3af3, 0x49b4,0x68a7
        DC16 0x0ee4,0x7f22, 0x7141,0x3ba5, 0x486a,0x698c
        DC16 0x0c8c,0x7f62, 0x70e3,0x3c57, 0x471d,0x6a6e
        DC16 0x0a33,0x7f98, 0x7083,0x3d08, 0x45cd,0x6b4b
        DC16 0x07d9,0x7fc2, 0x7023,0x3db8, 0x447b,0x6c24
        DC16 0x057f,0x7fe2, 0x6fc2,0x3e68, 0x4326,0x6cf9
        DC16 0x0324,0x7ff6, 0x6f5f,0x3f17, 0x41ce,0x6dca
        DC16 0x00c9,0x7fff, 0x6efb,0x3fc6, 0x4074,0x6e97
        DC16 0xfe6e,0x7ffe, 0x6e97,0x4074, 0x3f17,0x6f5f
        DC16 0xfc13,0x7ff1, 0x6e31,0x4121, 0x3db8,0x7023
        DC16 0xf9b8,0x7fd9, 0x6dca,0x41ce, 0x3c57,0x70e3
        DC16 0xf75e,0x7fb5, 0x6d62,0x427a, 0x3af3,0x719e
        DC16 0xf505,0x7f87, 0x6cf9,0x4326, 0x398d,0x7255
        DC16 0xf2ac,0x7f4e, 0x6c8f,0x43d1, 0x3825,0x7308
        DC16 0xf055,0x7f0a, 0x6c24,0x447b, 0x36ba,0x73b6
        DC16 0xedff,0x7eba, 0x6bb8,0x4524, 0x354e,0x7460
        DC16 0xebab,0x7e60, 0x6b4b,0x45cd, 0x33df,0x7505
        DC16 0xe958,0x7dfb, 0x6add,0x4675, 0x326e,0x75a6
        DC16 0xe707,0x7d8a, 0x6a6e,0x471d, 0x30fc,0x7642
        DC16 0xe4b9,0x7d0f, 0x69fd,0x47c4, 0x2f87,0x76d9
        DC16 0xe26d,0x7c89, 0x698c,0x486a, 0x2e11,0x776c
        DC16 0xe023,0x7bf9, 0x691a,0x490f, 0x2c99,0x77fb
        DC16 0xdddc,0x7b5d, 0x68a7,0x49b4, 0x2b1f,0x7885
        DC16 0xdb99,0x7ab7, 0x6832,0x4a58, 0x29a4,0x790a
        DC16 0xd958,0x7a06, 0x67bd,0x4afb, 0x2827,0x798a
        DC16 0xd71b,0x794a, 0x6747,0x4b9e, 0x26a8,0x7a06
        DC16 0xd4e1,0x7885, 0x66d0,0x4c40, 0x2528,0x7a7d
        DC16 0xd2ab,0x77b4, 0x6657,0x4ce1, 0x23a7,0x7aef
        DC16 0xd079,0x76d9, 0x65de,0x4d81, 0x2224,0x7b5d
        DC16 0xce4b,0x75f4, 0x6564,0x4e21, 0x209f,0x7bc6
        DC16 0xcc21,0x7505, 0x64e9,0x4ec0, 0x1f1a,0x7c2a
        DC16 0xc9fc,0x740b, 0x646c,0x4f5e, 0x1d93,0x7c89
        DC16 0xc7db,0x7308, 0x63ef,0x4ffb, 0x1c0c,0x7ce4
        DC16 0xc5c0,0x71fa, 0x6371,0x5098, 0x1a83,0x7d3a
        DC16 0xc3a9,0x70e3, 0x62f2,0x5134, 0x18f9,0x7d8a
        DC16 0xc198,0x6fc2, 0x6272,0x51cf, 0x176e,0x7dd6
        DC16 0xbf8c,0x6e97, 0x61f1,0x5269, 0x15e2,0x7e1e
        DC16 0xbd86,0x6d62, 0x616f,0x5303, 0x1455,0x7e60
        DC16 0xbb85,0x6c24, 0x60ec,0x539b, 0x12c8,0x7e9d
        DC16 0xb98b,0x6add, 0x6068,0x5433, 0x113a,0x7ed6
        DC16 0xb796,0x698c, 0x5fe4,0x54ca, 0x0fab,0x7f0a
        DC16 0xb5a8,0x6832, 0x5f5e,0x5560, 0x0e1c,0x7f38
        DC16 0xb3c0,0x66d0, 0x5ed7,0x55f6, 0x0c8c,0x7f62
        DC16 0xb1df,0x6564, 0x5e50,0x568a, 0x0afb,0x7f87
        DC16 0xb005,0x63ef, 0x5dc8,0x571e, 0x096b,0x7fa7
        DC16 0xae31,0x6272, 0x5d3e,0x57b1, 0x07d9,0x7fc2
        DC16 0xac65,0x60ec, 0x5cb4,0x5843, 0x0648,0x7fd9
        DC16 0xaaa0,0x5f5e, 0x5c29,0x58d4, 0x04b6,0x7fea
        DC16 0xa8e2,0x5dc8, 0x5b9d,0x5964, 0x0324,0x7ff6
        DC16 0xa72c,0x5c29, 0x5b10,0x59f4, 0x0192,0x7ffe
        DC16 0xa57e,0x5a82, 0x5a82,0x5a82, 0x0000,0x7fff
        DC16 0xa3d7,0x58d4, 0x59f4,0x5b10, 0xfe6e,0x7ffe
        DC16 0xa238,0x571e, 0x5964,0x5b9d, 0xfcdc,0x7ff6
        DC16 0xa0a2,0x5560, 0x58d4,0x5c29, 0xfb4a,0x7fea
        DC16 0x9f14,0x539b, 0x5843,0x5cb4, 0xf9b8,0x7fd9
        DC16 0x9d8e,0x51cf, 0x57b1,0x5d3e, 0xf827,0x7fc2
        DC16 0x9c11,0x4ffb, 0x571e,0x5dc8, 0xf695,0x7fa7
        DC16 0x9a9c,0x4e21, 0x568a,0x5e50, 0xf505,0x7f87
        DC16 0x9930,0x4c40, 0x55f6,0x5ed7, 0xf374,0x7f62
        DC16 0x97ce,0x4a58, 0x5560,0x5f5e, 0xf1e4,0x7f38
        DC16 0x9674,0x486a, 0x54ca,0x5fe4, 0xf055,0x7f0a
        DC16 0x9523,0x4675, 0x5433,0x6068, 0xeec6,0x7ed6
        DC16 0x93dc,0x447b, 0x539b,0x60ec, 0xed38,0x7e9d
        DC16 0x929e,0x427a, 0x5303,0x616f, 0xebab,0x7e60
        DC16 0x9169,0x4074, 0x5269,0x61f1, 0xea1e,0x7e1e
        DC16 0x903e,0x3e68, 0x51cf,0x6272, 0xe892,0x7dd6
        DC16 0x8f1d,0x3c57, 0x5134,0x62f2, 0xe707,0x7d8a
        DC16 0x8e06,0x3a40, 0x5098,0x6371, 0xe57d,0x7d3a
        DC16 0x8cf8,0x3825, 0x4ffb,0x63ef, 0xe3f4,0x7ce4
        DC16 0x8bf5,0x3604, 0x4f5e,0x646c, 0xe26d,0x7c89
        DC16 0x8afb,0x33df, 0x4ec0,0x64e9, 0xe0e6,0x7c2a
        DC16 0x8a0c,0x31b5, 0x4e21,0x6564, 0xdf61,0x7bc6
        DC16 0x8927,0x2f87, 0x4d81,0x65de, 0xdddc,0x7b5d
        DC16 0x884c,0x2d55, 0x4ce1,0x6657, 0xdc59,0x7aef
        DC16 0x877b,0x2b1f, 0x4c40,0x66d0, 0xdad8,0x7a7d
        DC16 0x86b6,0x28e5, 0x4b9e,0x6747, 0xd958,0x7a06
        DC16 0x85fa,0x26a8, 0x4afb,0x67bd, 0xd7d9,0x798a
        DC16 0x8549,0x2467, 0x4a58,0x6832, 0xd65c,0x790a
        DC16 0x84a3,0x2224, 0x49b4,0x68a7, 0xd4e1,0x7885
        DC16 0x8407,0x1fdd, 0x490f,0x691a, 0xd367,0x77fb
        DC16 0x8377,0x1d93, 0x486a,0x698c, 0xd1ef,0x776c
        DC16 0x82f1,0x1b47, 0x47c4,0x69fd, 0xd079,0x76d9
        DC16 0x8276,0x18f9, 0x471d,0x6a6e, 0xcf04,0x7642
        DC16 0x8205,0x16a8, 0x4675,0x6add, 0xcd92,0x75a6
        DC16 0x81a0,0x1455, 0x45cd,0x6b4b, 0xcc21,0x7505
        DC16 0x8146,0x1201, 0x4524,0x6bb8, 0xcab2,0x7460
        DC16 0x80f6,0x0fab, 0x447b,0x6c24, 0xc946,0x73b6
        DC16 0x80b2,0x0d54, 0x43d1,0x6c8f, 0xc7db,0x7308
        DC16 0x8079,0x0afb, 0x4326,0x6cf9, 0xc673,0x7255
        DC16 0x804b,0x08a2, 0x427a,0x6d62, 0xc50d,0x719e
        DC16 0x8027,0x0648, 0x41ce,0x6dca, 0xc3a9,0x70e3
        DC16 0x800f,0x03ed, 0x4121,0x6e31, 0xc248,0x7023
        DC16 0x8002,0x0192, 0x4074,0x6e97, 0xc0e9,0x6f5f
        DC16 0x8001,0xff37, 0x3fc6,0x6efb, 0xbf8c,0x6e97
        DC16 0x800a,0xfcdc, 0x3f17,0x6f5f, 0xbe32,0x6dca
        DC16 0x801e,0xfa81, 0x3e68,0x6fc2, 0xbcda,0x6cf9
        DC16 0x803e,0xf827, 0x3db8,0x7023, 0xbb85,0x6c24
        DC16 0x8068,0xf5cd, 0x3d08,0x7083, 0xba33,0x6b4b
        DC16 0x809e,0xf374, 0x3c57,0x70e3, 0xb8e3,0x6a6e
        DC16 0x80de,0xf11c, 0x3ba5,0x7141, 0xb796,0x698c
        DC16 0x812a,0xeec6, 0x3af3,0x719e, 0xb64c,0x68a7
        DC16 0x8181,0xec71, 0x3a40,0x71fa, 0xb505,0x67bd
        DC16 0x81e2,0xea1e, 0x398d,0x7255, 0xb3c0,0x66d0
        DC16 0x824f,0xe7cd, 0x38d9,0x72af, 0xb27f,0x65de
        DC16 0x82c6,0xe57d, 0x3825,0x7308, 0xb140,0x64e9
        DC16 0x8349,0xe330, 0x3770,0x735f, 0xb005,0x63ef
        DC16 0x83d6,0xe0e6, 0x36ba,0x73b6, 0xaecc,0x62f2
        DC16 0x846e,0xde9e, 0x3604,0x740b, 0xad97,0x61f1
        DC16 0x8511,0xdc59, 0x354e,0x7460, 0xac65,0x60ec
        DC16 0x85be,0xda18, 0x3497,0x74b3, 0xab36,0x5fe4
        DC16 0x8676,0xd7d9, 0x33df,0x7505, 0xaa0a,0x5ed7
        DC16 0x8738,0xd59e, 0x3327,0x7556, 0xa8e2,0x5dc8
        DC16 0x8805,0xd367, 0x326e,0x75a6, 0xa7bd,0x5cb4
        DC16 0x88dd,0xd134, 0x31b5,0x75f4, 0xa69c,0x5b9d
        DC16 0x89be,0xcf04, 0x30fc,0x7642, 0xa57e,0x5a82
        DC16 0x8aaa,0xccd9, 0x3042,0x768e, 0xa463,0x5964
        DC16 0x8ba0,0xcab2, 0x2f87,0x76d9, 0xa34c,0x5843
        DC16 0x8ca1,0xc890, 0x2ecc,0x7723, 0xa238,0x571e
        DC16 0x8dab,0xc673, 0x2e11,0x776c, 0xa129,0x55f6
        DC16 0x8ebf,0xc45b, 0x2d55,0x77b4, 0xa01c,0x54ca
        DC16 0x8fdd,0xc248, 0x2c99,0x77fb, 0x9f14,0x539b
        DC16 0x9105,0xc03a, 0x2bdc,0x7840, 0x9e0f,0x5269
        DC16 0x9236,0xbe32, 0x2b1f,0x7885, 0x9d0e,0x5134
        DC16 0x9371,0xbc2f, 0x2a62,0x78c8, 0x9c11,0x4ffb
        DC16 0x94b5,0xba33, 0x29a4,0x790a, 0x9b17,0x4ec0
        DC16 0x9603,0xb83c, 0x28e5,0x794a, 0x9a22,0x4d81
        DC16 0x9759,0xb64c, 0x2827,0x798a, 0x9930,0x4c40
        DC16 0x98b9,0xb462, 0x2768,0x79c9, 0x9843,0x4afb
        DC16 0x9a22,0xb27f, 0x26a8,0x7a06, 0x9759,0x49b4
        DC16 0x9b94,0xb0a2, 0x25e8,0x7a42, 0x9674,0x486a
        DC16 0x9d0e,0xaecc, 0x2528,0x7a7d, 0x9592,0x471d
        DC16 0x9e91,0xacfd, 0x2467,0x7ab7, 0x94b5,0x45cd
        DC16 0xa01c,0xab36, 0x23a7,0x7aef, 0x93dc,0x447b
        DC16 0xa1b0,0xa976, 0x22e5,0x7b27, 0x9307,0x4326
        DC16 0xa34c,0xa7bd, 0x2224,0x7b5d, 0x9236,0x41ce
        DC16 0xa4f0,0xa60c, 0x2162,0x7b92, 0x9169,0x4074
        DC16 0xa69c,0xa463, 0x209f,0x7bc6, 0x90a1,0x3f17
        DC16 0xa84f,0xa2c2, 0x1fdd,0x7bf9, 0x8fdd,0x3db8
        DC16 0xaa0a,0xa129, 0x1f1a,0x7c2a, 0x8f1d,0x3c57
        DC16 0xabcd,0x9f98, 0x1e57,0x7c5a, 0x8e62,0x3af3
        DC16 0xad97,0x9e0f, 0x1d93,0x7c89, 0x8dab,0x398d
        DC16 0xaf68,0x9c8f, 0x1cd0,0x7cb7, 0x8cf8,0x3825
        DC16 0xb140,0x9b17, 0x1c0c,0x7ce4, 0x8c4a,0x36ba
        DC16 0xb31f,0x99a9, 0x1b47,0x7d0f, 0x8ba0,0x354e
        DC16 0xb505,0x9843, 0x1a83,0x7d3a, 0x8afb,0x33df
        DC16 0xb6f1,0x96e6, 0x19be,0x7d63, 0x8a5a,0x326e
        DC16 0xb8e3,0x9592, 0x18f9,0x7d8a, 0x89be,0x30fc
        DC16 0xbadc,0x9448, 0x1833,0x7db1, 0x8927,0x2f87
        DC16 0xbcda,0x9307, 0x176e,0x7dd6, 0x8894,0x2e11
        DC16 0xbedf,0x91cf, 0x16a8,0x7dfb, 0x8805,0x2c99
        DC16 0xc0e9,0x90a1, 0x15e2,0x7e1e, 0x877b,0x2b1f
        DC16 0xc2f8,0x8f7d, 0x151c,0x7e3f, 0x86f6,0x29a4
        DC16 0xc50d,0x8e62, 0x1455,0x7e60, 0x8676,0x2827
        DC16 0xc727,0x8d51, 0x138f,0x7e7f, 0x85fa,0x26a8
        DC16 0xc946,0x8c4a, 0x12c8,0x7e9d, 0x8583,0x2528
        DC16 0xcb69,0x8b4d, 0x1201,0x7eba, 0x8511,0x23a7
        DC16 0xcd92,0x8a5a, 0x113a,0x7ed6, 0x84a3,0x2224
        DC16 0xcfbe,0x8972, 0x1073,0x7ef0, 0x843a,0x209f
        DC16 0xd1ef,0x8894, 0x0fab,0x7f0a, 0x83d6,0x1f1a
        DC16 0xd424,0x87c0, 0x0ee4,0x7f22, 0x8377,0x1d93
        DC16 0xd65c,0x86f6, 0x0e1c,0x7f38, 0x831c,0x1c0c
        DC16 0xd898,0x8637, 0x0d54,0x7f4e, 0x82c6,0x1a83
        DC16 0xdad8,0x8583, 0x0c8c,0x7f62, 0x8276,0x18f9
        DC16 0xdd1b,0x84d9, 0x0bc4,0x7f75, 0x822a,0x176e
        DC16 0xdf61,0x843a, 0x0afb,0x7f87, 0x81e2,0x15e2
        DC16 0xe1a9,0x83a6, 0x0a33,0x7f98, 0x81a0,0x1455
        DC16 0xe3f4,0x831c, 0x096b,0x7fa7, 0x8163,0x12c8
        DC16 0xe642,0x829d, 0x08a2,0x7fb5, 0x812a,0x113a
        DC16 0xe892,0x822a, 0x07d9,0x7fc2, 0x80f6,0x0fab
        DC16 0xeae4,0x81c1, 0x0711,0x7fce, 0x80c8,0x0e1c
        DC16 0xed38,0x8163, 0x0648,0x7fd9, 0x809e,0x0c8c
        DC16 0xef8d,0x8110, 0x057f,0x7fe2, 0x8079,0x0afb
        DC16 0xf1e4,0x80c8, 0x04b6,0x7fea, 0x8059,0x096b
        DC16 0xf43c,0x808b, 0x03ed,0x7ff1, 0x803e,0x07d9
        DC16 0xf695,0x8059, 0x0324,0x7ff6, 0x8027,0x0648
        DC16 0xf8ef,0x8032, 0x025b,0x7ffa, 0x8016,0x04b6
        DC16 0xfb4a,0x8016, 0x0192,0x7ffe, 0x800a,0x0324
        DC16 0xfda5,0x8006, 0x00c9,0x7fff, 0x8002,0x0192

        END
